;;;   lookup.el :: Looking up English/Japanese online dictionary
;;;
;;;		Copyright (C) 1994 Kazuhiko Yamamoto
;;;
;;;		   This emacs lisp library conforms
;;;		GNU GENERAL PUBLIC LICENSE Version 2.
;;;
;;; Author:  Kazuhiko Yamamoto <kazu@is.aist-nara.ac.jp>
;;; Created: December 26, 1994
;;; Revised: 
;;;

(defconst lookup-version "lookup.el version 3.00")

;;;
;;; (autoload 'lookup-word   "lookup" nil t)
;;; (autoload 'lookup-ispell "lookup" nil t)
;;; (autoload 'lookup-read   "lookup" nil t)
;;;

(defvar lookup-host-name "eri")
(defvar lookup-protocol-name "ndtp")

(defvar lookup-ispell-program "ispell")
(defvar lookup-ispell-version nil 
  "Set 2, 3 or nil. If nil, it is automatically guessed.")

(defvar lookup-ispell-args nil)
(defvar lookup-ispell-args-2 '("-a"))
(defvar lookup-ispell-args-3 '("-a" "-m"))

(defvar lookup-process nil)
(defvar lookup-ispell-process nil)
(defvar lookup-buffer         "*lookup*")
(defvar lookup-buffer-ispell " *lookup ispell*")
(defvar lookup-buffer-tmp    " *lookup tmp*")


(defvar lookup-filter
  '(("\\($B!%(B\\) \\([1-9][0-9]?[a(]? \\)" . "\\1\n\n  \\2")
    ("\\($B!%(B\\) \\([bcdefg] \\)" . "\\1\n  \\2")
    ("\015\005" . "\n  ")
    (" \\($B!](B[\\[ ]\\)" . "\n\n\n \\1"))
  )

(defun lookup-init ()
  (if (not (get-buffer lookup-buffer-ispell))
      (get-buffer-create lookup-buffer-ispell))
  (if (not (get-buffer lookup-buffer))
      (get-buffer-create lookup-buffer))
  (if (not (get-buffer lookup-buffer-tmp))
      (get-buffer-create lookup-buffer-tmp))
  (if (null lookup-ispell-version)
      (setq lookup-ispell-version (lookup-ispell-version)))
  (cond
   ((equal lookup-ispell-version 2)
    (or lookup-ispell-args (setq lookup-ispell-args lookup-ispell-args-2)))
   ((equal lookup-ispell-version 3)
    (or lookup-ispell-args (setq lookup-ispell-args lookup-ispell-args-3)))
   (t (error "Unsupported ispell version."))
   )
  )

(defun lookup-ispell-version ()
  "Rutern ispell version. If ispell is GNU ispell 4 or later, 
return nil since it don't return canonicalized words."
  (save-excursion
    (set-buffer lookup-buffer-ispell)
    (erase-buffer)
    (call-process lookup-ispell-program
		  "/dev/null" ;; to terminate GNU ispell
		  lookup-buffer-ispell
		  nil
		  "-v" ;; ignored by GNU ispell
		  )
    (goto-char (point-min))
    (let ((case-fold-search t))
      (if (re-search-forward "version \\([0-9]\\)\\." nil t nil)
	  (string-to-int (buffer-substring (match-beginning 1) (match-end 1))))
      )
    ))

(defun lookup-ispell-process-check ()
  (if (and lookup-ispell-process
	   (eq (process-status lookup-ispell-process) 'run))
      ()
    (message "Starting new ispell process ...")
    (setq lookup-ispell-process
	  (apply (function start-process)
		 "* lookup ispell *" ;; process name
		 lookup-buffer-ispell
		 lookup-ispell-program
		 lookup-ispell-args))
    (process-kill-without-query lookup-ispell-process)
    (sleep-for 3) ;; xxx
    )
  )


(defun lookup-extract-word ()
  (unwind-protect
      (save-excursion
	(if (not (looking-at "\\w"))
	    (re-search-backward "\\w" (point-min) t)) 
	(re-search-backward "\\W" (point-min) t)  
	(or (re-search-forward "\\w+" nil t)              
	    (error "No word to look up."))
	(buffer-substring (match-beginning 0) (match-end 0))
	)
    ))


(defun lookup-ispell-word (word)
  (save-excursion
    (set-buffer lookup-buffer-ispell)
    (erase-buffer)
    (lookup-ispell-process-check)
    (send-string lookup-ispell-process (concat word "\n"))
    (while (progn                 
	     (goto-char (point-max))
	     (/= (preceding-char) ?\n))
      (accept-process-output lookup-ispell-process))
    (goto-char (point-min))
    (lookup-ispell-parse-output
     (buffer-substring (point) (progn (end-of-line) (point))) word)
    ))

(defun lookup-ispell-parse-output (output word)
  (cond
   ((string-match "^+" output) (downcase (substring output 2)))
   (t (downcase word)) ;; "*" "#" "&"
   )
  )

(defun lookup-open-dictionary (dict)
  (save-excursion
    (set-buffer lookup-buffer-tmp)
    (if (and lookup-process
	     (eq (process-status lookup-process) 'open))
	()
      (setq lookup-process
	    (open-network-stream "*lookup dserver*"
				 lookup-buffer-tmp
				 lookup-host-name
				 lookup-protocol-name))
      (if (boundp 'MULE)
	  (define-program-coding-system nil lookup-process *euc-japan*)
;	  (set-process-coding-system lookup-process *internal* *euc-japan*)
	)
      (set-process-sentinel lookup-process 'lookup-sentinel)
      (erase-buffer)
      (process-send-string
       lookup-process (format "A%s@%s\n" (user-login-name) (system-name)))
      (if (null (string= 
		 "$A"
		 (lookup-get-answer '("$A" "$!" "$?" "$N") lookup-process)))
	  (error "Cannot connect to dserver"))
      )
    (erase-buffer)
    (process-send-string lookup-process (format "L%s\n" dict))
    (if (string=
	 "$*"
	 (lookup-get-answer '("$A" "$N" "$?" "$*" "$&") lookup-process))
	()
      (error "Cannot set up the dictionary"))
    ))


(defun lookup-sentinel (process event)
  (if (string= "finished\n" event)
      (setq lookup-process nil))
  )

(defun lookup-get-answer (answers process)
  (let ((regex (mapconcat (function regexp-quote) answers "\\|")))
    (save-excursion
      (set-buffer lookup-buffer-tmp)
      (sleep-for 2)
      (catch 'loop
	(while t
	  (goto-char (point-min))
	  (if (re-search-forward regex nil t)
	      (throw 'loop
		     (buffer-substring (match-beginning 0) (match-end 0))))
	  )
	(goto-char (point-max))
;	(accept-process-output)
	)
      )))


(defun lookup-lookup-dictionary (word)
  (let ((com nil))
    (cond
     ((car (find-charset-string word))
      (lookup-open-dictionary "waei")
      (setq com "Pk"))
     (t 
      (lookup-open-dictionary "eiwa")
      (setq com "Pa"))
     )
    (save-excursion
      (set-buffer lookup-buffer-tmp)
      (erase-buffer)
      (process-send-string lookup-process (format "%s%s\n" com word))
      (if (null (equal
		 "$$"
		 (lookup-get-answer
		  '("$$" "$!" "$?" "$N" "$A") lookup-process)))
	  (error "dserver don't know %s, sorry." (upcase word))
	(let ((entry))
	  (goto-char (point-min))
	  (re-search-forward (regexp-quote "$0"))
	  (forward-line)
	  (while (not (looking-at (regexp-quote "$$")))
	    (forward-line)
	    (setq entry (cons (buffer-substring
			       (point)
			       (progn (forward-line) (1- (point))))
			      entry))
	    )
	  (erase-buffer)
	  (setq entry (reverse entry))
	  (while entry
	    (process-send-string lookup-process (format "S%s\n" (car entry)))
	    (insert "\n")
	    (setq entry (cdr entry))
	    ))
	))
    ))

(defun lookup-word ()
  (interactive)
  (lookup-lookup-dictionary (lookup-extract-word))
  (lookup-show)
  )

(defun lookup-ispell ()
  (interactive)
  (let ((word (lookup-extract-word)))
     (if (null (car (find-charset-string word)))
	 (setq word (lookup-ispell-word word)))
     (lookup-lookup-dictionary (read-string "Look up: " word))
     )
  (lookup-show)
  )

(defun lookup-read ()
  (interactive)
  (lookup-lookup-dictionary (read-string "Look up: " ""))
  (lookup-show)
  )

(defun lookup-show ()
  (save-excursion
    (if (get-buffer-window lookup-buffer)
	()
      (delete-other-windows)
      (split-window)
      (display-buffer lookup-buffer (next-window (selected-window))))
    (set-buffer lookup-buffer)
    (goto-char (point-min))
    (erase-buffer)
    (sleep-for 1)
    (set-buffer lookup-buffer-tmp)
    (goto-char (point-min))
    (let (start end)
      (while (re-search-forward (regexp-quote "$1") nil t)
	(forward-line)
	(setq start (point))
	(forward-line)
	(setq end (point))
	;;
	(set-buffer lookup-buffer)
	(insert-buffer-substring lookup-buffer-tmp start end)
	(insert "\n*****\n")
	(set-buffer lookup-buffer-tmp)
	))
    (set-buffer lookup-buffer)
    (funcall 'lookup-reformatter)
    (funcall 'lookup-filter-buffer)
    )
  )

      
(defun lookup-reformatter ()
  (goto-char (point-min))
  (if (re-search-forward "\\[[^]]*\\]" nil t nil)
      (insert "\n\n"))
  )

(defun lookup-filter-buffer ()
  (goto-char (point-min))
  (let ((filter lookup-filter))
    (while filter
      (while (re-search-forward (car (car filter)) nil t nil)
	(replace-match (cdr (car filter)) t nil))
      (setq filter (cdr filter))
      )
    )
  )

(lookup-init)

;;
;; End of lookup.el
;;