! package derivative_mpdata
!     for 2-D thermal convection in a square box. 
!
! 99/05/05  S. Takehiro
! 99/05/21  S. Takehiro   double precision
! 99/08/20  S. Takehiro   separate from advect_mpdata
! 99/08/28  S. Takehiro   modify div
! 99/08/29  S. Takehiro   move div to module derivative
!
!======================================================================
module mpdata_gridtrans
  implicit none
  private

  public sg2vgx, sg2vgz, sg2vgxz, vgx2vgz, vgz2vgx

  integer :: nx, nz

  contains
   !------------------------------------------
    function sg2vgx(f)
      double precision, dimension(:,:),intent(in) :: f      ! scalar (on SG)
      double precision, &
           & dimension(0:size(f,1),size(f,2))     :: sg2vgx ! scalar (on VGX)

      nx = size(f,1) ; nz = size(f,2) 

      sg2vgx(1:nx-1,2:nz-1) &
        & =  (      f(1:nx-1,3:nz  ) +   f(2:nx,3:nz)      &
        &       + 2*f(1:nx-1,2:nz-1) + 2*f(2:nx,2:nz-1)    &
        &       +   f(1:nx-1,1:nz-2) +   f(2:nx,1:nz-2) ) / 8 

      sg2vgx(1:nx-1,1)  = ( f(1:nx-1,1)  +   f(2:nx,1) ) /2
!!$      sg2vgx(1:nx-1,1)  &
!!$        & =  (      f(1:nx-1,2) +   f(2:nx,2)    &
!!$        &       + 2*f(1:nx-1,1) + 2*f(2:nx,1)    &
!!$        &       +   f(1:nx-1,2) +   f(2:nx,2) ) / 8

      sg2vgx(1:nx-1,nz) = ( f(1:nx-1,nz) +   f(2:nx,nz) ) /2
!!$      sg2vgx(1:nx-1,nz) &
!!$        & =  (      f(1:nx-1,nz-1) +   f(2:nx,nz-1)  &
!!$        &       + 2*f(1:nx-1,nz)   + 2*f(2:nx,nz)    &
!!$        &       +   f(1:nx-1,nz-1) +   f(2:nx,nz-1) ) / 8


      sg2vgx(0,:)  = 0.0     ! ¿ʬ̵Ϥ(׳ǧ)
      sg2vgx(nx,:) = 0.0     ! 

    end function sg2vgx

   !------------------------------------------
    function sg2vgz(f)
      double precision, dimension(:,:),intent(in) :: f      ! scalar (on SG)
      double precision, &
           & dimension(size(f,1),0:size(f,2))     :: sg2vgz ! scalar (on VGX)

      nx = size(f,1) ; nz = size(f,2) 

      sg2vgz(2:nx-1,1:nz-1) &
        & =  (      f(3:nx,1:nz-1)   +   f(3:nx,2:nz)      &
        &       + 2*f(2:nx-1,1:nz-1) + 2*f(2:nx-1,2:nz)    &
        &       +   f(1:nx-2,1:nz-1) +   f(1:nx-2,2:nz) ) / 8 

      sg2vgz(1,1:nz-1)  = ( f(1,1:nz-1)  +   f(1,2:nz) ) /2
!!$      sg2vgz(1,1:nz-1) &
!!$        & =  (      f(2,1:nz-1) +   f(2,2:nz)    &
!!$        &       + 2*f(1,1:nz-1) + 2*f(1,2:nz)    &
!!$        &       +   f(2,1:nz-1) +   f(2,2:nz) ) / 8

      sg2vgz(nx,1:nz-1) = ( f(nx,1:nz-1) +   f(nx,2:nz) ) /2
!!$      sg2vgz(nx,1:nz-1) &
!!$        & =  (      f(nx-1,1:nz-1) +   f(nx-1,2:nz)  &
!!$        &       + 2*f(nx,1:nz-1) + 2*f(nx,2:nz)    &
!!$        &       +   f(nx-1,1:nz-1) +   f(nx-1,2:nz) ) / 8 

      sg2vgz(:,0)  = 0.0     ! ¿ʬ̵Ϥ(׳ǧ)
      sg2vgz(:,nz) = 0.0     ! 
    end function sg2vgz

   !------------------------------------------
    function sg2vgxz(f)
      double precision, dimension(:,:),intent(in) :: f       ! scalar (on SG)
      double precision, &
           & dimension(0:size(f,1),0:size(f,2))   :: sg2vgxz ! scalar (on VGX)

      nx = size(f,1) ; nz = size(f,2) 

      sg2vgxz(1:nx-1,1:nz-1) &
          & = (   f(2:nx,2:nz)   + f(2:nx,1:nz-1)  &
          &     + f(1:nx-1,2:nz) + f(1:nx-1,1:nz-1) ) / 4

      sg2vgxz(0,:) = 0.0 ; sg2vgxz(nx,:) = 0.0  ! ¿ʬ̵Ϥ(׳ǧ)
      sg2vgxz(:,0) = 0.0 ; sg2vgxz(:,nz) = 0.0

    end function sg2vgxz

   !------------------------------------------
    function vgx2vgz(u)
      double precision, dimension(0:,:),intent(in) :: u      ! flux field
                                                             !   (on VGX)
      double precision, &
           & dimension(size(u,1)-1,0:size(u,2))    :: vgx2vgz ! scalar (on VGZ)

      nx = size(u,1)-1 ; nz = size(u,2) 

      vgx2vgz(:,1:nz-1) &
          & = (   u(1:nx,2:nz)   + u(0:nx-1,2:nz)  &
          &     + u(1:nx,1:nz-1) + u(0:nx-1,1:nz-1) ) / 4

      vgx2vgz(:,0)  = 0.0    ! ¿ʬ̵Ϥ(׳ǧ)
      vgx2vgz(:,nz) = 0.0

    end function vgx2vgz

   !------------------------------------------
    function vgz2vgx(w)
      double precision, dimension(:,0:),intent(in) :: w      ! flux field
                                                             !   (on VGZ)
      double precision, &
           & dimension(0:size(w,1),size(w,2)-1)    :: vgz2vgx ! scalar (on VGZ)

      nx = size(w,1) ; nz = size(w,2)-1

      vgz2vgx(1:nx-1,:) &
          & = (   w(1:nx-1,1:nz)   + w(2:nx,1:nz) &
          &     + w(1:nx-1,0:nz-1) + w(2:nx,0:nz-1) ) / 4

      vgz2vgx(0,:)  = 0.0    ! ¿ʬ̵Ϥ(׳ǧ)
      vgz2vgx(nx,:) = 0.0

    end function vgz2vgx
end module mpdata_gridtrans

!======================================================================
module mpdata_derivatives
  use coordinates, only : dx, dz
  use timestep, only    : dt
  use mpdata_gridtrans
  use derivative
  implicit none

  integer :: nx, nz
  double precision   :: eps_mpdata = 1e-15      ! MPDATA 0 
  integer            :: mpdata_iteration=2      ! MPDATA ȿ

  private
  public div, flux_x_upstream, flux_z_upstream
  public flux_div_mpdata, u_mpdiff, w_mpdiff
  public eps_mpdata, mpdata_iteration

  contains
   !------------------------------------------
    function flux_x_upstream(f,u)
      double precision, dimension(:,:),     intent(in) :: f    ! scalar field
                                                               ! (on SG)
      double precision, & 
	& dimension(0:size(f,1),size(f,2)), intent(in) :: u    ! flow field
                                                               !   (on VGX)
      double precision, &
	& dimension(0:size(f,1),size(f,2)) :: flux_x_upstream  ! flux (on VGX)

      nx = size(f,1) ; nz = size(f,2) 
      flux_x_upstream(1:nx-1,:) &
          &  =   (  ( u(1:nx-1,:)+abs(u(1:nx-1,:)) )*f(1:nx-1,:) &
          &       + ( u(1:nx-1,:)-abs(u(1:nx-1,:)) )*f(2:nx,:) ) / 2.0

      flux_x_upstream(0,:)  = 0.0      ! ɤΥեå 0
      flux_x_upstream(nx,:) = 0.0      ! ɤΥեå 0
    end function flux_x_upstream

   !------------------------------------------
    function flux_z_upstream(f,w)
      double precision, dimension(:,:),     intent(in) :: f    ! scalar field
                                                               ! (on SG)
      double precision, & 
	& dimension(size(f,1),0:size(f,2)), intent(in) :: w    ! flow field
                                                               !   (on VGZ)
      double precision, &
	& dimension(size(f,1),0:size(f,2)) :: flux_z_upstream  ! flux (on VGZ)

      nx = size(f,1) ; nz = size(f,2) 

      flux_z_upstream(:,1:nz-1) &
          &  = ( ( w(:,1:nz-1)+abs(w(:,1:nz-1)) )*f(:,1:nz-1) & 
          &       + ( w(:,1:nz-1)-abs(w(:,1:nz-1)) )*f(:,2:nz) ) / 2.0

      flux_z_upstream(:,0)  = 0.0      ! ɤΥեå 0
      flux_z_upstream(:,nz) = 0.0      ! ɤΥեå 0
    end function flux_z_upstream

   !------------------------------------------
    function u_mpdiff(f,u,w)
      double precision, dimension(:,:), intent(in)        :: f ! scalar (on SG)
      double precision, &
           & dimension(0:size(f,1),size(f,2)), intent(in) :: u ! flow  (on VGX)
      double precision, &
           & dimension(size(f,1),0:size(f,2)), intent(in) :: w ! flow  (on VGZ)

      double precision, &
           & dimension(0:size(f,1),size(f,2))  :: u_mpdiff ! flow (on VGX)

      double precision, dimension(:,:), allocatable :: w_vgx  ! scalar (on VGX)
      double precision, dimension(:,:), allocatable :: f_vgx  ! scalar (on VGX)
      double precision, dimension(:,:), allocatable :: f_vgxz ! scalar (on VGXZ)
      nx = size(f,1) ; nz = size(f,2)

      allocate( w_vgx(0:nx,nz), f_vgx(0:nx,nz), f_vgxz(0:nx,0:nz) )

      w_vgx = vgz2vgx(w) ; f_vgx = sg2vgx(f) ; f_vgxz = sg2vgxz(f)
      !w_vgx(:,1) = 2*w_vgx(:,1) ; w_vgx(:,nz) = 2*w_vgx(:,nz)

      f_vgxz(0,:) = f_vgxz(1,:) ; f_vgxz(nx,:) = f_vgxz(nx-1,:)
      f_vgxz(:,0) = f_vgxz(:,1) ; f_vgxz(:,nz) = f_vgxz(:,nz-1)

      u_mpdiff(1:nx-1,:) = &
         & (   0.5*( abs(u(1:nx-1,:))*dx - dt*u(1:nx-1,:)**2) &
         &        *( f(2:nx,:)-f(1:nx-1,:) )/dx      &
         &   - 0.5*dt*u(1:nx-1,:)*w_vgx(1:nx-1,:)    &
         &       * ( f_vgxz(1:nx-1,1:nz) - f_vgxz(1:nx-1,0:nz-1) ) / dz ) &
         &   / ( f_vgx(1:nx-1,:) + eps_mpdata )

      u_mpdiff(0,:) = 0.0    ! ¿ʬ̵Ϥ(׳ǧ)
      u_mpdiff(nx,:) = 0.0

      deallocate( w_vgx, f_vgx, f_vgxz )
    end function u_mpdiff

   !------------------------------------------
    function w_mpdiff(f,u,w)
      double precision, dimension(:,:), intent(in)        :: f ! scalar (on SG)
      double precision, &
           & dimension(0:size(f,1),size(f,2)), intent(in) :: u ! flow  (on VGX)
      double precision, &
           & dimension(size(f,1),0:size(f,2)), intent(in) :: w ! flow  (on VGZ)

      double precision, &
           & dimension(size(f,1),0:size(f,2))  :: w_mpdiff ! flow (on VGX)

      double precision, dimension(:,:), allocatable :: u_vgz  ! scalar (on VGZ)
      double precision, dimension(:,:), allocatable :: f_vgz  ! scalar (on VGZ)
      double precision, dimension(:,:), allocatable :: f_vgxz ! scalar (on VGXZ)
      !double precision :: eps=1e-15

      nx = size(f,1) ; nz = size(f,2)

      allocate( u_vgz(nx,0:nz), f_vgz(nx,0:nz), f_vgxz(0:nx,0:nz) )

      u_vgz = vgx2vgz(u)  ; f_vgz = sg2vgz(f) ; f_vgxz = sg2vgxz(f)
      !u_vgz(1,:) = 2*u_vgz(1,:) ; u_vgz(:,nz) = 2*u_vgz(nz,:)

      !u_vgz(:,1) =0.0 ; u_vgz(:,nz) =0.0
      !f_vgz(:,0) = f_vgz(:,1) ; f_vgz(:,nz) = f_vgz(:,nz-2)
      f_vgxz(0,:) = f_vgxz(1,:) ; f_vgxz(nx,:) = f_vgxz(nx-1,:)
      f_vgxz(:,0) = f_vgxz(:,1) ; f_vgxz(:,nz) = f_vgxz(:,nz-1)

      w_mpdiff(:,1:nz-1) = &
         & (   0.5*( abs(w(:,1:nz-1))*dz - dt*w(:,1:nz-1)**2) &
         &        *( f(:,2:nz)-f(:,1:nz-1) )/dz      &
         &   - 0.5*dt*u_vgz(:,1:nz-1)*w(:,1:nz-1)    &
         &       * ( f_vgxz(1:nx,1:nz-1) - f_vgxz(0:nx-1,1:nz-1) ) / dx ) &
         &   / ( f_vgz(:,1:nz-1) + eps_mpdata )

      w_mpdiff(:,0) = 0.0    ! ¿ʬ̵Ϥ(׳ǧ)
      w_mpdiff(:,nz) = 0.0

      deallocate( u_vgz, f_vgz, f_vgxz )
    end function w_mpdiff

   !------------------------------------------
    function flux_div_mpdata(f,u,w)
      double precision, dimension(:,:), intent(in)        :: f ! scalar (on SG)
      double precision, &
           & dimension(0:size(f,1),size(f,2)), intent(in) :: u ! flow  (on VGX)
      double precision, &
           & dimension(size(f,1),0:size(f,2)), intent(in) :: w ! flow  (on VGZ)

      double precision, &
           & dimension(size(f,1),size(f,2))  :: flux_div_mpdata ! (on SG)

      double precision, &
           & dimension(:,:), allocatable :: fstar  ! temp. scalar (on SG)
      double precision, &
           & dimension(:,:), allocatable :: ustar  ! diff. vector (on VGX)
      double precision, &
           & dimension(:,:), allocatable :: wstar  ! diff. vector (on VGZ)
      double precision, &
           & dimension(:,:), allocatable :: ustar_new ! diff. vector (on VGX)
      double precision, &
           & dimension(:,:), allocatable :: wstar_new ! diff. vector (on VGZ)
      double precision, &
           & dimension(:,:), allocatable :: flux_x  ! flux (on VGX)
      double precision, &
           & dimension(:,:), allocatable :: flux_z  ! flux (on VGZ)

      integer             :: i

      nx = size(f,1) ; nz = size(f,2)

      allocate( fstar(nx,nz), ustar(0:nx,nz), wstar(nx,0:nz) )
      allocate( ustar_new(0:nx,nz), wstar_new(nx,0:nz) )
      allocate( flux_x(0:nx,nz), flux_z(nx,0:nz) )

      flux_x = flux_x_upstream(f,u)
      flux_z = flux_z_upstream(f,w)
      ustar = u ;  wstar = w

      do i=1, mpdata_iteration
        fstar = f - div(flux_x,flux_z) * dt

        ustar_new = u_mpdiff(fstar,ustar,wstar)
        wstar_new = w_mpdiff(fstar,ustar,wstar)

        ustar = ustar_new; wstar = wstar_new 

        flux_x = flux_x + flux_x_upstream(fstar,ustar)
        flux_z = flux_z + flux_z_upstream(fstar,wstar)
      end do

      flux_div_mpdata = div(flux_x,flux_z)
      
      deallocate( fstar, ustar, wstar )
      deallocate( ustar_new, wstar_new )
      deallocate( flux_x, flux_z )
    end function flux_div_mpdata

end module mpdata_derivatives
