! package dynamics
!     for 2-D thermal convection in a square box. 
!
! 99/05/04  S. Takehiro
! 99/05/21  S. Takehiro   double precision
! 99/05/25  S. Takehiro   separate jacobian
! 99/06/30  S. Takehiro   add diff_term
! 99/08/29  S. Takehiro   add div
!
!=======================================================================
module derivative

  use coordinates, only : dx, dz
  implicit none
  private

  public :: dxd, dzd, dxd2, dzd2, dxdz, laplace
  public :: div, j_arakawa, diff_term

  contains

 !------------------- $B4pK\HyJ,7W;;(B($B6-3&$G$NCM$O(B 0) ----------------
  function dxd(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: dxd
    integer                              :: nx

    nx = size(a,1)
    dxd = 0.0
    dxd(2:nx-1,:) = ( a(3:nx,:) - a(1:nx-2,:) ) / (2*dx)
  end function dxd

  function dzd(a)
    double precision, dimension(:,:)                  :: a
    double precision, dimension(size(a,1),size(a,2))  :: dzd
    integer                               :: nz

    nz = size(a,2)
    dzd = 0.0
    dzd(:,2:nz-1) = ( a(:,3:nz) - a(:,1:nz-2) ) / (2*dz)
  end function dzd

  function dxd2(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: dxd2
    integer                              :: nx

    nx = size(a,1)
    dxd2 = 0.0
    dxd2(2:nx-1,:) = ( a(1:nx-2,:) - 2*a(2:nx-1,:) + a(3:nx,:) ) / dx**2
  end function dxd2

  function dxdz(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: dxdz
    integer                              :: nx, nz

    nx = size(a,1)  ;   nz = size(a,2)
    dxdz = 0.0
    dxdz(2:nx-1,2:nz-1) &
       & = ( a(3:nx,3:nz) -a(1:nx-2,3:nz) -a(3:nx,1:nz-2) +a(1:nx-2,1:nz-2) ) &
       &    / (4*dx*dz)
  end function dxdz

  function dzd2(a)
    double precision, dimension(:,:)                  :: a
    double precision, dimension(size(a,1),size(a,2))  :: dzd2
    integer                               :: nz

    nz = size(a,2)
    dzd2 = 0.0
    dzd2(:,2:nz-1) = ( a(:,1:nz-2) - 2*a(:,2:nz-1) + a(:,3:nz) ) / dz**2
  end function dzd2

  function laplace(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: laplace

    laplace = dxd2(a) + dzd2(a)
  end function laplace

 !-------------------- $BH/;6(B (divergence ) ----------------------
  function div(flux_x,flux_z)
    double precision, dimension(0:,:),intent(in)    :: flux_x  ! flux field 
                                                               !    (on VGX)
    double precision, dimension(:,0:),intent(in)    :: flux_z  ! flux field
                                                               !    (on VGZ)
    double precision, &
         & dimension(size(flux_z,1),size(flux_x,2)) :: div     ! scalor field

    integer                                         :: nx, nz

    nx = size(flux_z,1) ; nz = size(flux_x,2) 

   !--- $BFbItNN0h7W;;(B

    div(2:nx-1,2:nz-1) = &
         &    ( flux_x(2:nx-1,2:nz-1) - flux_x(1:nx-2,2:nz-1) )/dx &
         &  + ( flux_z(2:nx-1,2:nz-1) - flux_z(2:nx-1,1:nz-2) )/dz

   !--- $B6-3&$G$OMWAGLL@QH>J,(B, $B30It$+$i$N%U%i%C%/%9(B 0

    div(1,2:nz-1) = &    ! $B:8C<(B
         &   ( flux_x(1,2:nz-1) - 0.0 ) / (dx/2) &
         & + ( flux_z(1,2:nz-1) - flux_z(1,1:nz-2) )/dz

    div(nx,2:nz-1) = &   ! $B1&C<(B
         &   ( 0.0 - flux_x(nx-1,2:nz-1) ) / (dx/2) &
         & + ( flux_z(nx,2:nz-1) - flux_z(nx,1:nz-2) )/dz

    div(2:nx-1,1) = &    ! $B2<C<(B
         &   ( flux_x(2:nx-1,1) - flux_x(1:nx-2,1) )/dx &
         & + ( flux_z(2:nx-1,1) - 0.0 ) / (dz/2)

    div(2:nx-1,nz) = &   ! $B>eC<(B
         &   ( flux_x(2:nx-1,nz) - flux_x(1:nx-2,nz) )/dx &
         & + ( 0.0 - flux_z(2:nx-1,nz-1)) / (dz/2)

    div(1,1) = &    ! $B:82<6y(B
         &   ( flux_x(1,1) - 0.0 ) / (dx/2) &
         & + ( flux_z(1,1) - 0.0 ) / (dz/2) 

    div(nx,1) = &   ! $B1&2<6y(B
         &   ( 0.0 - flux_x(nx-1,1) ) / (dx/2) &
         & + ( flux_z(nx,1) - 0.0 ) / (dz/2) 

    div(1,nz) = &   ! $B:8>e6y(B
         &   ( flux_x(1,nz) - 0.0 ) / (dx/2) &
         & + ( 0.0 - flux_z(1,nz-1) ) / (dz/2) 

    div(nx,nz) = &   ! $B1&2<6y(B
         &   ( 0.0 - flux_x(nx-1,nz) ) / (dx/2) &
         & + ( 0.0 - flux_z(nx,nz-1) ) / (dz/2) 

  end function div

 !---------------------- $B0\N.9`(B(Arakawa Jacobian) ---------------------
  function j_arakawa(a,b)
    double precision, dimension(:,:)                 :: a,b
    double precision, dimension(size(a,1),size(a,2)) :: j_arakawa

    integer                              :: nx, nz

    nx = size(a,1) ; nz = size(a,2)

    j_arakawa = 0.0

    j_arakawa( 2:nx-1,2:nz-1 ) &
     &   =(       ( a(3:nx,2:nz-1)-a(1:nx-2,2:nz-1) )/(2*dx)   &
     &          * ( b(2:nx-1,3:nz)-b(2:nx-1,1:nz-2) )/(2*dz)   &
     &        -   ( a(2:nx-1,3:nz)-a(2:nx-1,1:nz-2) )/(2*dz)   &
     &          * ( b(3:nx,2:nz-1)-b(1:nx-2,2:nz-1) )/(2*dx)   &
     &     +      ( b(2:nx-1,3:nz)*( a(3:nx,3:nz)-a(1:nx-2,3:nz) )  &
     &              - b(2:nx-1,1:nz-2)*( a(3:nx,1:nz-2)-a(1:nx-2,1:nz-2) )) &
     &               / (4*dx*dz) &
     &      -     ( b(3:nx,2:nz-1)*( a(3:nx,3:nz)-a(3:nx,1:nz-2) )  &
     &              - b(1:nx-2,2:nz-1)*( a(1:nx-2,3:nz)-a(1:nx-2,1:nz-2) )) &
     &               / (4*dx*dz) &
     &     +      ( a(3:nx,2:nz-1)*( b(3:nx,3:nz)-b(3:nx,1:nz-2) )  &
     &              - a(1:nx-2,2:nz-1)*( b(1:nx-2,3:nz)-b(1:nx-2,1:nz-2) )) &
     &               / (4*dx*dz) &
     &        -   ( a(2:nx-1,3:nz)*( b(3:nx,3:nz)-b(1:nx-2,3:nz) )  &
     &              - a(2:nx-1,1:nz-2)*( b(3:nx,1:nz-2)-b(1:nx-2,1:nz-2) )) &
     &               / (4*dx*dz) &
     &         )/3.0

  end function j_arakawa

 !-------------------------- $B3H;69`(B --------------------------
  function diff_term( diff_x, diff_z, a )
    double precision, dimension(:,:)                   :: a
    double precision, dimension(size(a,1)-1,size(a,2)) :: diff_x
    double precision, dimension(size(a,1),size(a,2)-1) :: diff_z
    double precision, dimension(size(a,1),size(a,2))   :: diff_term

    double precision, dimension(:,:), allocatable  :: flux_x, flux_z

    integer :: nx, nz

    nx = size(a,1) ; nz = size(a,2)

    allocate( flux_x(0:nx,nz), flux_z(nx,0:nz) )
                          ! $B6-3&$r1[$($?H>@0?tE@$G$b%U%i%C%/%9$rDj5A(B
                          ! $B6-3&$G%U%i%C%/%9(B 0 $B$G$N3H;69`$r7W;;(B

    flux_x = 0.0d0 ; flux_z = 0.0d0

    flux_x(1:nx-1,:) = - diff_x(1:nx-1,:) * ( a(2:nx,:) - a(1:nx-1,:) )/dx
    flux_z(:,1:nz-1) = - diff_z(:,1:nz-1) * ( a(:,2:nz) - a(:,1:nz-1) )/dz

    diff_term = - div( flux_x, flux_z )

    deallocate( flux_x, flux_z )
  end function diff_term

end module derivative
