! package dynamics
!     for 2-D thermal convection in a square box. 
!
! 99/05/04  S. Takehiro
! 99/05/21  S. Takehiro   double precision
! 99/05/25  S. Takehiro   separate jacobian
! 99/06/30  S. Takehiro   add diff_term
! 99/08/29  S. Takehiro   add div
! 99/09/26  S. Takehiro   cyclic b.c. 
!
!=======================================================================
module derivative_base

  use coordinates, only : dx, dz, xcycl, zcycl
  implicit none
  private

  public :: dxd, dzd, dxd2, dzd2, dxdz, laplace

  contains

 !------------------- $B4pK\HyJ,7W;;(B($B6-3&$G$NCM$O(B 0) ----------------
  function dxd(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: dxd
    integer                              :: nx

    nx = size(a,1)
    dxd = 0.0
    dxd(2:nx-1,:) = ( a(3:nx,:) - a(1:nx-2,:) ) / (2*dx)
    if ( xcycl ) then 
       dxd(nx,:)     = ( a(2,:) - a(nx-1,:) ) / (2*dx)
       dxd(1,: )     = ( a(2,:) - a(nx-1,:) ) / (2*dx)
    endif

  end function dxd

  function dzd(a)
    double precision, dimension(:,:)                  :: a
    double precision, dimension(size(a,1),size(a,2))  :: dzd
    integer                               :: nz

    nz = size(a,2)
    dzd = 0.0
    dzd(:,2:nz-1) = ( a(:,3:nz) - a(:,1:nz-2) ) / (2*dz)

    if ( zcycl ) then 
       dzd(:,nz) = ( a(:,2) - a(:,nz-1) ) / (2*dz)
       dzd(:,1) = ( a(:,2) - a(:,nz-1) )  / (2*dz)
    endif

  end function dzd

  function dxd2(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: dxd2
    integer                              :: nx

    nx = size(a,1)
    dxd2 = 0.0
    dxd2(2:nx-1,:) = ( a(1:nx-2,:) - 2*a(2:nx-1,:) + a(3:nx,:) ) / dx**2

    if ( xcycl ) then
       dxd2(nx,:)     = ( a(2,:) - 2*a(1,:) + a(nx-1,:) ) / dx**2
       dxd2(1,:)      = ( a(2,:) - 2*a(1,:) + a(nx-1,:) ) / dx**2
    endif

  end function dxd2

  function dxdz(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: dxdz
    integer                              :: nx, nz

    nx = size(a,1)  ;   nz = size(a,2)
    dxdz = 0.0

    dxdz(2:nx-1,2:nz-1) &
       & = ( a(3:nx,3:nz) -a(1:nx-2,3:nz) -a(3:nx,1:nz-2) +a(1:nx-2,1:nz-2) ) &
       &    / (4*dx*dz)

    if ( xcycl ) then
       dxdz(nx,2:nz-1) &
            & = ( a(2,3:nz) -a(nx-1,3:nz) -a(2,1:nz-2) +a(nx-1,1:nz-2) ) &
            &    / (4*dx*dz)
       dxdz(1,2:nz-1) &
            & = ( a(2,3:nz) -a(nx-1,3:nz) -a(2,1:nz-2) +a(nx-1,1:nz-2) ) &
            &    / (4*dx*dz)
    endif
    if ( zcycl ) then
       dxdz(2:nx-1,nz) &
            & = ( a(3:nx,2) -a(1:nx-2,2) -a(3:nx,nz-1) +a(1:nx-2,nz-1) ) &
            &    / (4*dx*dz)
       dxdz(2:nx-1,1) &
            & = ( a(3:nx,2) -a(1:nx-2,2) -a(3:nx,nz-1) +a(1:nx-2,nz-1) ) &
            &    / (4*dx*dz)
    endif

  end function dxdz

  function dzd2(a)
    double precision, dimension(:,:)                  :: a
    double precision, dimension(size(a,1),size(a,2))  :: dzd2
    integer                               :: nz

    nz = size(a,2)
    dzd2 = 0.0
    dzd2(:,2:nz-1) = ( a(:,1:nz-2) - 2*a(:,2:nz-1) + a(:,3:nz) ) / dz**2

    if ( zcycl ) then
       dzd2(:,nz) = ( a(:,nz-1) - 2*a(:,nz) + a(:,2) ) / dz**2
       dzd2(:,1)  = ( a(:,nz-1) - 2*a(:,1)  + a(:,2) ) / dz**2
    endif

  end function dzd2

  function laplace(a)
    double precision, dimension(:,:)                 :: a
    double precision, dimension(size(a,1),size(a,2)) :: laplace

    laplace = dxd2(a) + dzd2(a)
  end function laplace

end module derivative_base
