! package averages
!     for 2-D thermal convection in a square box. 
!
! 99/08/24  S. Takehiro
! 
!======================================================================
module average_base

  use coordinates, only : dx, dz, xcycl, zcycl
  implicit none
  private

  public :: avr_x_1d, avr_x_2d, avr_z_1d, avr_z_2d, avr_xz

  contains

    function avr_x_1d(a)
      double precision :: a(:)
      double precision :: avr_x_1d
      integer :: nx
      nx = size(a)

      if ( xcycl ) then
         avr_x_1d = sum(a(1:nx-1)) * dx
      else
         avr_x_1d =  a(1) * dx/2 + sum(a(2:nx-1)) * dx + a(nx) * dx/2 
      endif
    end function avr_x_1d

    function avr_x_2d(a)
      double precision :: a(:,:)
      double precision :: avr_x_2d(size(a,2))
      integer :: i, nx, nz
      nx = size(a,1) ; nz = size(a,2)

      if ( xcycl ) then
         do i = 1, nz
            avr_x_2d(i) = sum(a(1:nx-1,i)) * dx
         enddo
      else
         do i = 1, nz
            avr_x_2d(i) &
             & = a(1,i) * dx/2 + sum(a(2:nx-1,i)) * dx + a(nx,i) * dx/2 
         enddo
      endif
    end function avr_x_2d

    function avr_z_1d(a)
      double precision :: a(:)
      double precision :: avr_z_1d
      integer :: nz
      nz = size(a)

      if ( zcycl ) then
         avr_z_1d =  sum( a(1:nz-1) ) * dz
      else
         avr_z_1d = a(1) * dz/2 + sum(a(2:nz-1)) * dz + a(nz) * dz/2 
      endif
    end function avr_z_1d

    function avr_z_2d(a)
      double precision :: a(:,:)
      double precision :: avr_z_2d(size(a,1))

      integer :: i, nx, nz
      nx = size(a,1) ; nz = size(a,2)

      if ( zcycl ) then
         do i = 1, nx
            avr_z_2d(i) =  sum( a(i,1:nz-1) ) * dz
         enddo
      else
         do i = 1, nx
            avr_z_2d(i) &
               & =  a(i,1) * dz/2 + sum(a(i,2:nz-1)) * dz + a(i,nz) * dz/2 
         enddo
      endif
    end function avr_z_2d

    double precision function avr_xz(a)
      double precision :: a(:,:)

      avr_xz = avr_z_1d( avr_x_2d(a) )

    end function avr_xz


  end module average_base

!======================================================================
module average

  use average_base

  public :: avr_x, avr_z, avr_xz

  interface avr_x
     module procedure avr_x_1d, avr_x_2d
  end interface

  interface avr_z
     module procedure avr_z_1d, avr_z_2d
  end interface

  interface avr_xz
     module procedure avr_xz
  end interface

end module average

