#!perl -w
# Copyright (C) 2020-2021 all contributors <meta@public-inbox.org>
# License: AGPL-3.0+ <https://www.gnu.org/licenses/agpl-3.0.txt>
use strict;
use Test::More;
use PublicInbox::TestCommon;
use PublicInbox::MsgIter qw(msg_part_text);
my @classes = qw(PublicInbox::Eml);
SKIP: {
	require_mods('Email::MIME', 1);
	# TODO: Email::MIME behavior is not consistent in newer versions
	# we need to evaluate and possibly adjust our behavior to decide
	# between DWIM-ness with historical mail...
	push @classes, 'PublicInbox::MIME';
};
use_ok $_ for @classes;

sub mime_load ($) {
	my ($path) = @_;
	open(my $fh, '<', $path) or die "open $path: $!";
	PublicInbox::MIME->new(\(do { local $/; <$fh> }));
}

{
	my $eml = PublicInbox::Eml->new(\(my $str = "a: b\n\nhi\n"));
	is($str, "hi\n", '->new modified body like Email::Simple');
	is($eml->body, "hi\n", '->body works');
	is($eml->as_string, "a: b\n\nhi\n", '->as_string');
	my $empty = PublicInbox::Eml->new("\n\n");
	is($empty->as_string, "\n\n", 'empty message');
}

for my $cls (@classes) {
	my $mime = $cls->new(my $orig = "From: x\n\nb");
	is($mime->as_string, $orig, '->as_string works');
	is($mime->header_obj->as_string, "From: x\n",
			'header ->as_string works');

	# headers
	is($mime->header_raw('From'), 'x', 'header_raw scalar context');
	$mime = $cls->new("R:\n\tx\nR:\n 1\n");
	is_deeply([$mime->header_raw('r')], [ 'x', '1' ], 'multi-value');
	$mime = $cls->new("R:x\nR: 1\n");
	is_deeply([$mime->header_raw('r')], [ 'x', '1' ], 'multi-value header');
	$mime = $cls->new("R:x\n R: 1\nR:\n f\n");
	is_deeply([$mime->header_raw('r')], [ 'x R: 1', 'f' ],
		'multi-line, multi-value header');

	$mime->header_set('r');
	is_deeply([$mime->header_raw('r')], [], 'header_set clears');
	$mime->header_set('r');
	is_deeply([$mime->header_raw('r')], [], 'header_set clears idempotent');
	$mime->header_set('r', 'h');
	is_deeply([$mime->header_raw('r')], ['h'], 'header_set');
	$mime->header_set('r', 'h', 'i');
	is_deeply([$mime->header_raw('r')], ['h', 'i'], 'header_set ary');
	$mime->header_set('rr', 'b');
	is_deeply([$mime->header_raw('r')], ['h', 'i'],
				"header_set `rr' did not clobber `r'");
	is($mime->header_raw('rr'), 'b', 'got set scalar');
	$mime->header_set('rr', 'b'x100);
	is($mime->header_raw('rr'), 'b'x100, 'got long set scalar');
	if ($cls eq 'PublicInbox::Eml') {
		like($mime->as_string, qr/^rr: b{100}\n(?:\n|\z)/sm,
			'single token not wrapped');
	}
	$mime->header_set('rr', ('b'x100) . ' wrap me');
	if ($cls eq 'PublicInbox::Eml') {
		like($mime->as_string, qr/^rr: b{100}\n\twrap me\n/sm,
			'wrapped after long token');
	}
	my $exp = "pre\tformatted\n with\n breaks";
	$mime->header_set('r', $exp);
	like($mime->as_string, qr/^r: \Q$exp\E/sm, 'preformatted preserved');
} # for @classes

for my $cls (@classes) { # make sure we don't add quotes if not needed
	my $eml = $cls->new("From: John Smith <j\@example.com>\n\n");
	is($eml->header('From'), 'John Smith <j@example.com>',
		"name not unnecessarily quoted $cls");
}

for my $cls (@classes) {
	my $eml = $cls->new("Subject: foo\n\n");
	$eml->header_str_set('Subject', "\x{100}");
	like($eml->header_raw('Subject'), qr/utf-8\?B\?/i,
		'MIME-B encoded UTF-8 Subject');
	is_deeply([$eml->header('Subject')], [ "\x{100}" ],
		'got wide character back');
}

# linux-mips apparently got some messages injected w/o Message-ID
# and long Subject: lines w/o leading whitespace.
# What appears in the blobs was generated by V2Writable.
for my $cls (@classes) {
	my $eml = $cls->new(<<'EOF');
Message-ID: <20101130193431@z>
Subject: something really long
and really wrong
From: linux-mips archive injection
Object-Id: 8c56b7abdd551b1264e6522ededbbed9890cccd0
EOF
	is_deeply([ $eml->header('Subject') ],
		[ 'something really long and really wrong' ],
		'continued long line w/o leading spaces '.$cls);
	is_deeply([ $eml->header('From') ],
		[ 'linux-mips archive injection' ],
		'subsequent line not corrupted');
	is_deeply([ $eml->header('Message-ID') ],
		['<20101130193431@z>'],
		'preceding line readable');
} # for @classes

{
	my $eml = eml_load 't/msg_iter-order.eml';
	my @parts;
	my $orig = $eml->as_string;
	$eml->each_part(sub {
		my ($part, $level, @ex) = @{$_[0]};
		my $s = $part->body_str;
		$s =~ s/\s+//sg;
		push @parts, [ $s, $level, @ex ];
	});
	is_deeply(\@parts, [ [ qw(a 1 1) ], [ qw(b 1 2) ] ], 'order is fine');
	is($eml->as_string, $orig, 'unchanged by ->each_part');
	$eml->each_part(sub {}, undef, 1);
	is(defined($eml) ? $eml->body_raw : '', # old msg_iter clobbers $eml
		'', 'each_part can clobber body');
}

if ('descend into message/rfc822') {
	my $eml = eml_load 't/data/message_embed.eml';
	my @parts;
	$eml->each_part(sub {
		my ($part, $level, @ex) = @{$_[0]};
		push @parts, [ $part, $level, @ex ];
	});
	is(scalar(@parts), 6, 'got all parts');
	like($parts[0]->[0]->body, qr/^testing embedded message harder\n/sm,
		'first part found');
	is_deeply([ @{$parts[0]}[1..2] ], [ 1, '1' ],
		'got expected depth and level for part #0');
	is($parts[1]->[0]->filename, 'embed2x.eml',
		'attachment filename found');
	is_deeply([ @{$parts[1]}[1..2] ], [ 1, '2' ],
		'got expected depth and level for part #1');
	is_deeply([ @{$parts[2]}[1..2] ], [ 2, '2.1' ],
		'got expected depth and level for part #2');
	is_deeply([ @{$parts[3]}[1..2] ], [ 3, '2.1.1' ],
		'got expected depth and level for part #3');
	is_deeply([ @{$parts[4]}[1..2] ], [ 3, '2.1.2' ],
		'got expected depth and level for part #4');
	is($parts[4]->[0]->filename, 'test.eml',
		'another attachment filename found');
	is_deeply([ @{$parts[5]}[1..2] ], [ 4, '2.1.2.1' ],
		'got expected depth and level for part #5');
}

# body-less, boundary-less
for my $cls (@classes) {
	my $call = 0;
	$cls->new(<<'EOF')->each_part(sub { $call++ }, 0, 1);
Content-Type: multipart/mixed; boundary="body-less"

EOF
	is($call, 1, 'called on bodyless multipart');

	my @tmp;
	$cls->new(<<'EOF')->each_part(sub { push @tmp, \@_; }, 0, 1);
Content-Type: multipart/mixed; boundary="boundary-less"

hello world
EOF
	is(scalar(@tmp), 1, 'got one part even w/o boundary');
	is($tmp[0]->[0]->[0]->body, "hello world\n", 'body preserved');
	is($tmp[0]->[0]->[1], 0, '$depth is zero');
	is($tmp[0]->[0]->[2], 1, '@idx is one');
}

# I guess the following only worked in PI::M because of a happy accident
# involving inheritance:
for my $cls (@classes) {
	my @tmp;
	my $header_less = <<'EOF';
Archived-At: <85k5su9k59.fsf_-_@lola.goethe.zz>
Content-Type: multipart/mixed; boundary="header-less"

--header-less

this is the body

--header-less
i-haz: header

something else

--header-less--
EOF
	my $expect = "this is the body\n";
	$cls->new($header_less)->each_part(sub { push @tmp, \@_  }, 0, 1);
	my $body = $tmp[0]->[0]->[0]->body;
	if ($cls eq 'PublicInbox::Eml') {
		is($body, $expect, 'body-only subpart in '.$cls);
	} elsif ($body ne $expect) {
		diag "W: $cls `$body' != `$expect'";
	}
	is($tmp[1]->[0]->[0]->body, "something else\n");
	is(scalar(@tmp), 2, 'two parts');
}

if ('one newline before headers') {
	my $eml = PublicInbox::Eml->new("\nNewline: no Header \n");
	my @v = $eml->header_raw('Newline');
	is_deeply(\@v, ['no Header'], 'no header');
	is($eml->crlf, "\n", 'got CRLF as "\n"');
	is($eml->body, "");
}

if ('body only') {
	my $str = <<EOM;
--- a/lib/PublicInbox/Eml.pm
+++ b/lib/PublicInbox/Eml.pm
@@ -122,9 +122,10 @@ sub new {
\x20
EOM
	my $eml = PublicInbox::Eml->new($str);
	is($eml->body, $str, 'body-only accepted');
}

for my $cls (@classes) { # XXX: matching E::M, but not sure about this
	my $s = <<EOF;
Content-Type: multipart/mixed; boundary="b"

--b
header: only
--b--
EOF
	my $eml = $cls->new(\$s);
	my $nr = 0;
	my @v;
	$eml->each_part(sub {
		@v = $_[0]->[0]->header_raw('Header');
		$nr++;
	});
	is($nr, 1, 'only one part');
	is_deeply(\@v, [], "nothing w/o body $cls");
}

for my $cls (@classes) {
	my $s = <<EOF; # double epilogue, double the fun
Content-Type: multipart/mixed; boundary="b"

--b
should: appear

yes

--b--

--b
should: not appear

nope
--b--
EOF
	my $eml = $cls->new(\$s);
	my $nr = 0;
	$eml->each_part(sub {
		my $part = $_[0]->[0];
		is_deeply([$part->header_raw('should')], ['appear'],
			'only got one header');
		is($part->body, "yes\n", 'got expected body');
		$nr++;
	});
	is($nr, 1, 'only one part');
}

for my $cls (@classes) {
SKIP: {
	skip 'newer Email::MIME behavior inconsistent', 1 if
		$cls eq 'PublicInbox::MIME';
	my $s = <<EOF; # buggy git-send-email versions, again?
Content-Type: text/plain; =?ISO-8859-1?Q?=20charset=3D=1BOF?=
Content-Transfer-Encoding: 8bit
Object-Id: ab0440d8cd6d843bee9a27709a459ce3b2bdb94d (lore/kvm)

\xc4\x80
EOF
	my $eml = $cls->new(\$s);
	my ($str, $err) = msg_part_text($eml, $eml->content_type);
	is($str, "\x{100}\n", "got wide character by assuming utf-8 ($cls)");
} # SKIP
}

if ('we differ from Email::MIME with final "\n" on missing epilogue') {
	my $s = <<EOF;
Content-Type: multipart/mixed; boundary="b"

--b
header: but

no epilogue
EOF
	my $eml = PublicInbox::Eml->new(\$s);
	is(($eml->subparts)[-1]->body, "no epilogue\n",
		'final "\n" preserved on missing epilogue');
}

if ('header_size_limit stolen from postfix') {
	local $PublicInbox::Eml::header_size_limit = 4;
	my @w;
	local $SIG{__WARN__} = sub { push @w, @_ };
	my $eml = PublicInbox::Eml->new("a:b\na:d\n\nzz");
	is_deeply([$eml->header('a')], ['b'], 'no overrun header');
	is($eml->body_raw, 'zz', 'body not damaged');
	is($eml->header_obj->as_string, "a:b\n", 'header truncated');
	is(grep(/truncated/, @w), 1, 'truncation warned');

	$eml = PublicInbox::Eml->new("a:b\na:d\n");
	is_deeply([$eml->header('a')], ['b'], 'no overrun header w/o body');

	local $PublicInbox::Eml::header_size_limit = 5;
	$eml = PublicInbox::Eml->new("a:b\r\na:d\r\n\nzz");
	is_deeply([$eml->header('a')], ['b'], 'no overrun header on CRLF');
	is($eml->body_raw, 'zz', 'body not damaged');

	@w = ();
	$eml = PublicInbox::Eml->new("too:long\n");
	$eml = PublicInbox::Eml->new("too:long\n\n");
	$eml = PublicInbox::Eml->new("too:long\r\n\r\n");
	is(grep(/ignored/, @w), 3, 'ignored header warned');
}

if ('maxparts is a feature unique to us') {
	my $eml = eml_load 't/psgi_attach.eml';
	my @orig;
	$eml->each_part(sub { push @orig, $_[0]->[0] });

	local $PublicInbox::Eml::mime_parts_limit = scalar(@orig);
	my $i = 0;
	$eml->each_part(sub {
		my $cur = $_[0]->[0];
		my $prv = $orig[$i++];
		is($cur->body_raw, $prv->body_raw, "part #$i matches");
	});
	is($i, scalar(@orig), 'maxparts honored');
	$PublicInbox::Eml::mime_parts_limit--;
	my @ltd;
	$eml->each_part(sub { push @ltd, $_[0]->[0] });
	for ($i = 0; $i <= $#ltd; $i++) {
		is($ltd[$i]->body_raw, $orig[$i]->body_raw,
			"part[$i] matches");
	}
	is(scalar(@ltd), scalar(@orig) - 1, 'maxparts honored');
}

SKIP: {
	require_mods('PublicInbox::MIME', 1);
	my $eml = eml_load 't/utf8.eml';
	my $mime = mime_load 't/utf8.eml';
	for my $h (qw(Subject From To)) {
		my $v = $eml->header($h);
		my $m = $mime->header($h);
		is($v, $m, "decoded -8 $h matches Email::MIME");
		ok(utf8::is_utf8($v), "$h is UTF-8");
		ok(utf8::valid($v), "UTF-8 valid $h");
	}
	my $s = $eml->body_str;
	ok(utf8::is_utf8($s), 'body_str is UTF-8');
	ok(utf8::valid($s), 'UTF-8 valid body_str');
	my $ref = \(my $x = 'ref');
	for my $msg ($eml, $mime) {
		$msg->body_str_set($s .= "\nHI\n");
		ok(!utf8::is_utf8($msg->body_raw),
				'raw octets after body_str_set');
		$s = $msg->body_str;
		ok(utf8::is_utf8($s), 'body_str is UTF-8 after set');
		ok(utf8::valid($s), 'UTF-8 valid body_str after set');
		$msg->body_set($ref);
		is($msg->body_raw, $$ref, 'body_set worked on scalar ref');
		$msg->body_set($$ref);
		is($msg->body_raw, $$ref, 'body_set worked on scalar');
	}
	$eml = eml_load 't/iso-2202-jp.eml';
	$mime = mime_load 't/iso-2202-jp.eml';
	$s = $eml->body_str;
	is($s, $mime->body_str, 'ISO-2202-JP body_str');
	ok(utf8::is_utf8($s), 'ISO-2202-JP => UTF-8 body_str');
	ok(utf8::valid($s), 'UTF-8 valid body_str');

	$eml = eml_load 't/psgi_attach.eml';
	$mime = mime_load 't/psgi_attach.eml';
	is_deeply([ map { $_->body_raw } $eml->subparts ],
		[ map { $_->body_raw } $mime->subparts ],
		'raw ->subparts match deeply');
	is_deeply([ map { $_->body } $eml->subparts ],
		[ map { $_->body } $mime->subparts ],
		'->subparts match deeply');
	for my $msg ($eml, $mime) {
		my @old = $msg->subparts;
		$msg->parts_set([]);
		is_deeply([$msg->subparts], [], 'parts_set can clear');
		$msg->parts_set([$old[-1]]);
		is(scalar $msg->subparts, 1, 'only last remains');
	}

	# some versions of Email::MIME or Email::MIME::* will drop
	# unnecessary ", while PublicInbox::Eml will preserve the original
	my $exp = $mime->as_string;
	$exp =~ s/; boundary=b\b/; boundary="b"/;
	is($eml->as_string, $exp, 'as_string matches after parts_set');
}

for my $cls (@classes) {
	my $s = <<'EOF';
Content-Type: text/x-patch; name="=?utf-8?q?vtpm-fakefile.patch?="
Content-Disposition: attachment; filename="=?utf-8?q?vtpm-makefile.patch?="

EOF
	is($cls->new($s)->filename, 'vtpm-makefile.patch',
		"filename decoded ($cls)") if $cls ne 'PublicInbox::MIME';
	$s =~ s/^Content-Disposition:.*$//sm;
	is($cls->new($s)->filename, 'vtpm-fakefile.patch',
		"filename fallback ($cls)") if $cls ne 'PublicInbox::MIME';
	is($cls->new($s)->content_type,
		'text/x-patch; name="vtpm-fakefile.patch"',
		qq[matches Email::MIME output, "correct" or not ($cls)]);

	$s = <<'EOF';
Content-Type: multipart/foo; boundary=b

--b
Content-Disposition: attachment; filename="=?utf-8?q?vtpm-makefile.patch?="

a
--b
Content-Type: text/x-patch; name="=?utf-8?q?vtpm-fakefile.patch?="

b
--b--
EOF
	SKIP: {
		skip 'newer Email::MIME is inconsistent here', 1
			if $cls eq 'PublicInbox::MIME';
		my @x;
		$cls->new($s)->each_part(sub { push @x, $_[0]->[0]->filename });
		is_deeply(['vtpm-makefile.patch', 'vtpm-fakefile.patch'], \@x,
			"got filename for both attachments ($cls)");
	}
}

done_testing;
