%feature("docstring") OT::PointConditionalDistribution
R"RAW(Distribution conditioned by a deterministic vector.

.. warning::
    This class is experimental and likely to be modified in future releases.
    To use it, import the ``openturns.experimental`` submodule.

Let :math:`\inputRV` be a random vector of dimension :math:`\inputDim`. Let :math:`\cI \subset \{1, \dots, \inputDim \}`
be a set of indices of components of  :math:`\inputRV`, :math:`\overline{\cI}` its complementary in
:math:\{1, \dots, \inputDim \}` and :math:`\vect{x}_\cI` a real vector of dimension equal to the cardinal of :math:`\cI`.
Then this class implements the distribution of:

.. math::

    \inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI


This class requires the following features:

- each component :math:`X_i` is continuous or discrete: e.g., it can not be a :class:`~openturns.Mixture` of
  discrete and continuous distributions,
- the copula of :math:`\inputRV` is continuous: e.g., it can not be the :class:`~openturns.MinCopula`,
- the random vector :math:`\inputRV_{\overline{\cI}}` is continuous or discrete: all its components must be discrete
  or all its components must be continuous,
- the random vector :math:`\inputRV_{\cI}` may have some discrete components and some continuous components. 

Then, the pdf (probability density function if :math:`\inputRV_{\overline{\cI}}` is continuous or probability distribution function if
:math:`\inputRV_{\overline{\cI}}` is discrete) of :math:`\inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI` is defined by (in the following expression, we assumed a particular order of the conditioned components among the whole set of components for easy reading):

.. math::
    :label: condPDF

    p_{\inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI}(x_{\overline{\cI}})  = \dfrac{p_{\inputRV}(\vect{x
    }_{\overline{\cI}}, \vect{x}_{\cI})}{p_{\inputRV_{\cI}}(\vect{x}_{\cI})}

where:

.. math::

    p_{\inputRV}(\vect{x})  = \left( \prod_{i=1}^\inputDim p_i(x_i)\right) c(F_1(x_1), \dots,
    F_\inputDim(x_\inputDim))

with:

- :math:`c` is the probability density copula of :math:`\inputRV`,
- if :math:`X_i` is a continuous component, :math:`p_i` is its probability density function,
- if :math:`X_i` is a discrete component, :math:`p_i = \sum_{x^i_k \in \cS^i} \Prob{X_i = x^i_k} \delta_{x^i_k}` where
  :math:`\cS^i = \{ x^i_k \}` is its support and :math:`\delta_{x^i_k}` the Dirac distribution centered on
  :math:`x^i_k`.

Then, if :math:`\inputRV_{\overline{\cI}}` is continuous, we have:

.. math::

    p_{\inputRV_{\cI}}(\vect{x}_{\cI})  = \int p_{\inputRV}(\vect{x}_{\overline{\cI}}, \vect{x}_{\cI}) \,
    \di{\vect{x}_{\overline{\cI}}}

and if :math:`\inputRV_{\overline{\cI}}` is discrete with its support denoted by :math:`\cS(\vect{X}_{\overline{\cI}}) = \prod_{i \in \overline{\cI}} S^i`, we have:

.. math::

    p_{\inputRV_{\cI}}(\vect{x}_{\cI})  & = \sum_{\vect{x}_{\overline{\cI}} \in \cS(\inputRV_{\overline{\cI}})}
    p_{\inputRV}(\vect{x}_{\overline{\cI}}, \vect{x}_{\cI})

Parameters
----------
distribution : :class:`~openturns.Distribution`,
    Distribution of :math:`\inputRV`.
conditioningIndices : sequence of int,
    Conditioning vector indices :math:`\cI`.
conditioningValues : sequence of float,
    Conditioning vector values :math:`\vect{x}_\cI`.

Notes
-----
Simplification mechanisms to compute :eq:`condPDF` are implemented for some distributions.
To know if a simplification is used, ask the method *hasSimplifiedVersion*.

If the dimension of :math:`\inputRV_\cI` is greater than 1, the default integration algorithm is :class:`~openturns.CubaIntegration`
defined in the entry *PointConditionalDistribution-CubaAlgorithm* if available and the
:class:`~openturns.SimplicialCubature` algorithm if not. If the
dimension of :math:`\inputRV_\cI` is 1, then the default integration algorithm is the :class:`~openturns.GaussKronrod` algorithm with its default
quadrature formula.

In the case where no simplification mechanism has been implemented for the distribution of :math:`\inputRV`, then the range of
the conditioned distribution is computed according to one of the following strategies. The default strategy is defined in the
:class:`~openturns.ResourceMap`, entry *PointConditionalDistribution-RangeAdaptationMethod*.


**Strategy None**: The numerical range of :math:`\inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI` is the same as the
numerical range of :math:`\inputRV_{\overline{\cI}}`.
This range is exact for all distributions with bounded support. For distributions with unbounded support,
it is potentially false when the conditional values are very close to the bounds of the initial numerical support.


**Strategy Normal**: Let  :math:`\vect{Y}` be the Gaussian vector of dimension :math:`\inputDim`, which mean vector
is defined by :math:`\vect{\mu} = \Expect{\inputRV}` and covariance matrix is defined by :math:`\mat{C} = \Cov{\inputRV}`.
Then, we build the conditioned Gaussian vector:

.. math::

    \vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{x}_\cI


The numerical range :math:`\cD \left(\vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{x}_\cI \right)` of
:math:`\vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{x}_\cI` is known exactly thanks to the simplification mechanism
implemented for Gaussian vectors. We assign to :math:`\inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI` the
range :math:`\cD \left(\vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{x}_\cI \right)`:

.. math::

    \cD\left(\inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI \right) = \cD \left(\vect{Y}_{\overline{\cI}}|\vect{Y}_\cI
    = \vect{x}_\cI \right)


**Strategy NormalCopula**: Let  :math:`\vect{Y}` be the Gaussian vector of dimension :math:`\inputDim`, with zero mean,
unit variance and which correlation matrix :math:`\mat{R}` is defined from the Spearman correlation matrix of :math:`\inputRV`:
:math:`\left( \rho_S(X_i, X_j) \right)_{1 \leq i, j \leq \inputDim}`. Thus, :math:`\vect{Y}` is the standard representant
of the normal copula having the same correlation as :math:`\inputRV`.

For each conditioning value :math:`x_i`, we define the quantile :math:`q_i` of the normal distribution with zero mean and unit
variance associated to the same order as :math:`x_i`, for :math:`i \in \cI`:

.. math::

    q_i & = \Phi^{-1} \circ F_i \left (x_i \right)

where :math:`\Phi` is the CDF of the normal distribution with zero mean and unit variance. Then, we build the conditioned
Gaussian vector:

.. math::

    \vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{q}_\cI

which numerical range :math:`\cD\left(\vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{q}_\cI \right)` can be exactly computed.
Let it be:

.. math::

    \cD\left(\vect{Y}_{\overline{\cI}}|\vect{Y}_\cI = \vect{q}_\cI \right) = \prod_{i\in \overline{\cI}}
    \left[ y_i^{min}, y_i^{max}\right]

Then, inversely, we compute the quantiles of each :math:`F_i` for :math:`i \in \cI` which have the same order as the bounds
:math:`y_i^{min}` and :math:`y_i^{max}` with respect :math:`\Phi`:

.. math::

    x_i^{min} & = F_i^{-1}\circ \Phi \left (y_i^{min} \right) \\
    x_i^{max} & = F_i^{-1}\circ \Phi \left (y_i^{max} \right)

We assign to :math:`\inputRV_{\overline{\cI}}|\inputRV_\cI = \vect{x}_\cI` the numerical range defined by:

.. math::

    \cD \left(\vect{X}_{\overline{\cI}}|\vect{X}_\cI = \vect{x}_\cI \right) = \prod_{i\in \overline{\cI}} \left[ x_i^{min},
    x_i^{max}\right]


Several :class:`~openturns.ResourceMap` entries allow for a more fine-grained control over the distribution:

- The boolean entry *PointConditionalDistribution-UseSimplifiedVersion* decides if simplification is enabled.
- The string entry *PointConditionalDistribution-CubaAlgorithm* defines the default Cuba algorithm used to compute the CDF when the conditioning 
  values are mutivariate.
- The string entry *PointConditionalDistribution-OptimizationAlgorithm* defines the default optimization algorithm name used.
- The integer entry *PointConditionalDistribution-RatioUniformCandidateNumber* sets the number
  of candidates to find a feasible points for the ratio of uniforms method initialization.
- The boolean entry *PointConditionalDistribution-InitializeSampling* decides whether to initialize the ratio of uniforms method.
- The integer entry *PointConditionalDistribution-SmallDimension* is the dimension threshold
  from which to switch to CDF inversion sampling instead of ratio of uniforms (for continuous distributions).
- The boolean entry *PointConditionalDistribution-InitializeTransformation* decides whether
  to initialize the iso-probabilistic transformation.
- The boolean entry *PointConditionalDistribution-UseGenericConditionalMethods* decides whether
  the generic conditional methods are used (see :meth:`~openturns.Distribution.computeConditionalCDF()` for example). These methods can only be
  used when we want to create the distribution of :math:`X_d|(X_1, \dots, X_{d-1}) = (x_1, \dots, x_{d-1})` or when :math:`\inputRV` is bivariate
  with a symmetric copula.

Examples
--------
>>> import openturns as ot
>>> import openturns.experimental as otexp
>>> indices = [1]
>>> values = [2.0]
>>> normal = ot.Normal(3)
>>> R = ot.CorrelationMatrix(3)
>>> R[1, 0] = R[2, 0] = R[2, 1] = 0.9
>>> normal.setR(R)
>>> distribution = otexp.PointConditionalDistribution(normal, indices, values)
>>> sample = distribution.getSample(2)
)RAW"

// ----------------------------------------------------------------------------

%feature("docstring") OT::PointConditionalDistribution::getDistribution
R"RAW(Underlying distribution accessor.

Returns
-------
distribution : :class:`~openturns.Distribution`
    Distribution :math:`\cL_\theta(\inputReal)`)RAW"

// ----------------------------------------------------------------------------

%feature("docstring") OT::PointConditionalDistribution::getConditioningIndices
R"RAW(Conditioning vector indices accessor.

Returns
-------
conditioningIndices : :class:`~openturns.Indices`
    Conditioning vector indices :math:`\cI`)RAW"

// ----------------------------------------------------------------------------

%feature("docstring") OT::PointConditionalDistribution::getConditioningValues
R"RAW(Conditioning vector values accessor.

Returns
-------
conditioningValues : :class:`~openturns.Point`
    Conditioning vector values :math:`\vect{x}_\cI`)RAW"

// ----------------------------------------------------------------------------

%feature("docstring") OT::PointConditionalDistribution::getSimplifiedVersion
"Simplified distribution accessor.

Returns
-------
simplified : :class:`~openturns.Distribution`
    Simplified distribution.

Notes
-----
When a simplified version of the conditional exists, it avoids to use the generic methods:
it is more efficient and more precise."

// ---------------------------------------------------------------------

%feature("docstring") OT::PointConditionalDistribution::getIntegrationAlgorithm
R"RAW(Accessor to the integration algorithm used to compute the CDF.

Returns
-------
algo : :class:`~openturns.IntegrationAlgorithm`
    Integration algorithm used to compute the CDF.

Notes
-----
By default, if the dimension of :math:`\inputRV_\cI` is greater than 1, the integration algorithm is :class:`~openturns.CubaIntegration`
defined in the entry `PointConditionalDistribution-CubaAlgorithm` if available and the :class:`~openturns.SimplicialCubature` algorithm if not. If the
dimension of :math:`\inputRV_\cI` is 1, then the integration algorithm is the :class:`~openturns.GaussKronrod` algorithm with ts default quadrature
formula.)RAW"
// ---------------------------------------------------------------------

%feature("docstring") OT::PointConditionalDistribution::setIntegrationAlgorithm
"Accessor to the integration algorithm used to compute the CDF.

Parameters
----------
algo : :class:`~openturns.IntegrationAlgorithm`
    Integration algorithm used to compute the CDF."
