// Copyright 2014 The Go Authors. All rights reserved.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

//go:build ignore

// Generate Windows callback assembly file.
// This was copied from the Go repository and modified to generate non-Windows assembly files.

package main

import (
	"bytes"
	"fmt"
	"os"
)

const maxCallback = 2000

func genasm386() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build linux

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOVL and JMP instructions.
// The MOVL instruction loads CX with the callback index, and the
// JMP instruction branches to callbackasm1.
// callbackasm1 takes the callback index from CX and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVL $%d, CX\n", i)
		buf.WriteString("\tJMP  callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_386.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmAmd64() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// runtime·callbackasm is called by external code to
// execute Go implemented callback function. It is not
// called from the start, instead runtime·compilecallback
// always returns address into runtime·callbackasm offset
// appropriately so different callbacks start with different
// CALL instruction in runtime·callbackasm. This determines
// which Go callback function is executed later on.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		buf.WriteString("\tCALL callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_amd64.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmArm() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build linux

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOVW and B instructions.
// The MOVW instruction loads R12 with the callback index, and the
// B instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R12 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVW $%d, R12\n", i)
		buf.WriteString("\tB    callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_arm.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmArm64() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOV and B instructions.
// The MOV instruction loads R12 with the callback index, and the
// B instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R12 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVD $%d, R12\n", i)
		buf.WriteString("\tB    callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_arm64.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmLoong64() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOVV and JMP instructions.
// The MOVV instruction loads R12 with the callback index, and the
// JMP instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R12 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVV $%d, R12\n", i)
		buf.WriteString("\tJMP  callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_loong64.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmPpc64le() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build linux

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOVD and BR instructions.
// The MOVD instruction loads R11 with the callback index, and the
// BR instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R11 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVD $%d, R11\n", i)
		buf.WriteString("\tBR   callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_ppc64le.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmRiscv64() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build darwin || freebsd || linux || netbsd

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOV and JMP instructions.
// Since Go 1.26, MOV instructions with immediate values lower than or equal to 32
// are encoded in 2 bytes rather than 4 bytes, which breaks the assumption that each
// callback entry is 8 bytes long. Therefore, for callback indices less than or equal to 32,
// add a PCALIGN directive to align the next instruction to an 8-byte boundary.
// The MOV instruction loads X7 with the callback index, and the
// JMP instruction branches to callbackasm1.
// callbackasm1 takes the callback index from X7 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		space := " "
		if i <= 32 {
			fmt.Fprintf(&buf, "\tPCALIGN $8\n")
			space = "     "
		}
		fmt.Fprintf(&buf, "\tMOV%s$%d, X7\n", space, i)
		fmt.Fprintf(&buf, "\tJMP%scallbackasm1(SB)\n", space)
	}
	if err := os.WriteFile("zcallback_riscv64.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func genasmS390x() {
	var buf bytes.Buffer

	buf.WriteString(`// Code generated by wincallback.go using 'go generate'. DO NOT EDIT.

//go:build linux

// External code calls into callbackasm at an offset corresponding
// to the callback index. Callbackasm is a table of MOVD and BR instructions.
// The MOVD instruction loads R0 with the callback index, and the
// BR instruction branches to callbackasm1.
// callbackasm1 takes the callback index from R0 and
// indexes into an array that stores information about each callback.
// It then calls the Go implementation for that callback.
// NOTE: We use R0 instead of R11 because R11 is callee-saved on S390X.
#include "textflag.h"

TEXT callbackasm(SB), NOSPLIT|NOFRAME, $0
`)
	for i := 0; i < maxCallback; i++ {
		fmt.Fprintf(&buf, "\tMOVD $%d, R0\n", i)
		buf.WriteString("\tBR   callbackasm1(SB)\n")
	}
	if err := os.WriteFile("zcallback_s390x.s", buf.Bytes(), 0644); err != nil {
		fmt.Fprintf(os.Stderr, "wincallback: %s\n", err)
		os.Exit(2)
	}
}

func main() {
	genasm386()
	genasmAmd64()
	genasmArm()
	genasmArm64()
	genasmLoong64()
	genasmPpc64le()
	genasmRiscv64()
	genasmS390x()
}
