static char rcsid[] = "$Id: extension-search.c 223762 2020-12-14 04:54:10Z twu $";
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_MEMCPY
#define memcpy(d,s,n) bcopy((s),(d),(n))
#endif

#include "extension-search.h"

#ifdef WORDS_BIGENDIAN
#define CONVERT(x) Bigendian_convert_uint(x)
#include "bigendian.h"
#else
#define CONVERT(x) (x)
#include "littleendian.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/mman.h>		/* For munmap */

#include "mem.h"
#include "bool.h"
#include "assert.h"
#include "access.h"
#include "types.h"
#include "univcoord.h"
#include "oligo.h"

#include "list.h"
#include "maxent_hr.h"

#include "genome128_hr.h"
#include "genome128_consec.h"

#include "univdiagdef.h"
#include "univdiag.h"
#include "path-solve.h"

#if defined(HAVE_SSE2)
#include <emmintrin.h>
#endif
#ifdef HAVE_SSE4_1
#include <smmintrin.h>
#endif
#ifdef HAVE_AVX2
#include <immintrin.h>
#endif
#ifdef HAVE_AVX512
#include <immintrin.h>
#endif


/* conservative behaves better for high-quality sequences and is faster */
/* #define LIBERAL 1 */
/* #define LIBERAL_KMER 8 */

/* Some limit is needed to prevent GSNAP from running very slowly */
#define MAX_HITS_FOR_BEST_ELT 1000

#define MAX_INDEX1INTERVAL 6


/* #define CHECK_OLIGOS 1 */

#ifdef DEBUG
#define debug(x) x
#else
#define debug(x)
#endif

/* binary_search */
#ifdef DEBUG10
#define debug10(x) x
#else
#define debug10(x)
#endif

/* Sorting of diagonals */
#ifdef DEBUG12
#define debug12(x) x
#else
#define debug12(x)
#endif

/* filter_elts */
#ifdef DEBUG13
#define debug13(x) x
#else
#define debug13(x)
#endif


static Mode_T mode;

static Genome_T genomebits;
static Genome_T genomebits_alt;

static Univ_IIT_T chromosome_iit;
static Univcoord_T genomelength;
static int circular_typeint;
static bool *circularp;

static Genome_T genomebits;
static Genome_T genomebits_alt;
static Indexdb_T indexdb;
static Indexdb_T indexdb2;


static char conversion_fwd[128];
static char conversion_rev[128];

static int index1part;
static int index1interval;
static int local1part;
static int leftreadshift;
static Oligospace_T oligobase_mask;


#ifdef LARGE_GENOMES
#define GETPOS(high,low) (((Univcoord_T) high << 32) + low)
#endif


void
Extension_search_setup (Mode_T mode_in, Univ_IIT_T chromosome_iit_in,
			Univcoord_T genomelength_in, int circular_typeint_in, bool *circularp_in,
			Genome_T genomebits_in, Genome_T genomebits_alt_in, Indexdb_T indexdb_in, Indexdb_T indexdb2_in,
			int index1part_in, int index1interval_in, int local1part_in) {
  int i;

  mode = mode_in;

  chromosome_iit = chromosome_iit_in;
  genomelength = genomelength_in;
  circular_typeint = circular_typeint_in;
  circularp = circularp_in;

  genomebits = genomebits_in;
  genomebits_alt = genomebits_alt_in;
  indexdb = indexdb_in;
  indexdb2 = indexdb2_in;

  genomebits = genomebits_in;
  genomebits_alt = genomebits_alt_in;

  for (i = 0; i < 128; i++) {
    conversion_fwd[i] = i;
    conversion_rev[i] = i;
  }
  if (mode == STANDARD) {
    /* Don't change conversion */
  } else if (mode == CMET_STRANDED || mode == CMET_NONSTRANDED) {
    conversion_fwd['C'] = 'T';	/* CT */
    conversion_rev['G'] = 'A';	/* GA */
  } else if (mode == ATOI_STRANDED || mode == ATOI_NONSTRANDED) {
    conversion_fwd['A'] = 'G';	/* AG */
    conversion_rev['T'] = 'C';	/* TC */
  } else if (mode == TTOC_STRANDED || mode == TTOC_NONSTRANDED) {
    conversion_fwd['T'] = 'C';	/* TC */
    conversion_rev['A'] = 'G';	/* AG */
  }

  index1part = index1part_in;
  index1interval = index1interval_in;
  local1part = local1part_in;

#ifdef HAVE_64_BIT
  leftreadshift = 64 - index1part - index1part;
  oligobase_mask = ~(~ (Oligospace_T) 0 << 2*index1part);
#else
  leftreadshift = 32 - index1part - index1part;
  oligobase_mask = ~(~ (Oligospace_T) 0 << 2*index1part);
#endif

  return;
}


/* Simplified version of Spanningelt_T */
#define T Elt_T


#if 0
/* Works only for queryfwd.  Use Elt_read functions instead */
static T
Elt_new (int querypos, int nmatches, Univcoord_T *all_diagonals, int n_all_diagonals) {
  T new = (T) MALLOC(sizeof(*new));
#ifdef DEBUG
  int i;
#endif

  new->qstart = querypos;
  new->qend = querypos + nmatches - 1;
  new->nmatches = nmatches;
  debug(printf("Making an Elt with querystart %d, nmatches %d => queryend %d\n",
	       new->qstart,new->nmatches,new->qend));

  new->all_diagonals = all_diagonals;
  new->n_all_diagonals = n_all_diagonals;

  new->diagonals = &(new->all_diagonals[0]);
  new->ndiagonals = n_all_diagonals;

#ifdef DEBUG
  printf("Diagonals:");
  for (i = 0; i < n_all_diagonals; i++) {
    printf(" %llu",all_diagonals[i]);
  }
  printf("\n");
#endif

  new->lowi = 0;
  new->highi = n_all_diagonals;

  return new;
}
#endif


static void
Elt_free (T *old) {

  FREE((*old)->all_univdiagonals);
  FREE(*old);
  return;
}

void
Elt_gc (List_T *set) {
  List_T p;
  T elt;

  for (p = *set; p != NULL; p = List_next(p)) {
    elt = (T) List_head(p);
    Elt_free(&elt);
  }
  /* List_free(&(*set)); -- allocated by Listpool_push */
  return;
}



#if 0
static int
nt_querylength (char *query, int querylength) {
  int i;
  char c;

  i = 0;
  while (i < querylength && ((c = query[i]) == 'A' || c == 'C' || c == 'G' || c == 'T')) {
    i++;
  }

  return i;
}
#endif

#ifdef CHECK_OLIGOS
static Oligospace_T
nt_oligo (char *query, int indexsize) {
  Oligospace_T oligo = 0U;
  int i;

  for (i = 0; i < indexsize; i++) {
    oligo *= 4;
    
    printf("%c",query[i]);
    switch (query[i]) {
    case 'A': break;
    case 'C': oligo += 1; break;
    case 'G': oligo += 2; break;
    case 'T': oligo += 3; break;
    default:
      fprintf(stderr,"Saw N in nt_oligo\n");
      abort();
    }
  }
  printf("\n");

  return oligo;
}

#define LOW_TWO_BITS 0x3


static char *
oligo_nt (UINT4 oligo, int oligosize) {
  char *nt = MALLOC((oligosize+1)*sizeof(char));
  int i, j;
  UINT4 lowbits;

  j = oligosize-1;
  for (i = 0; i < oligosize; i++) {
    lowbits = oligo & LOW_TWO_BITS;
    switch (lowbits) {
    case RIGHT_A: nt[j] = 'A'; break;
    case RIGHT_C: nt[j] = 'C'; break;
    case RIGHT_G: nt[j] = 'G'; break;
    case RIGHT_T: nt[j] = 'T'; break;
    }
    oligo >>= 2;
    j--;
  }
  nt[oligosize] = '\0';

  return nt;
}

#endif


#if !defined(HAVE_SSE4_2)
#define count_leading_zeroes_32(diff) ((diff >> 16) ? clz_table[diff >> 16] : 16 + clz_table[diff])
#elif defined(HAVE_LZCNT)
#define count_leading_zeroes_32(diff) _lzcnt_u32(diff)
#elif defined(HAVE_BUILTIN_CLZ)
#define count_leading_zeroes_32(diff) __builtin_clz(diff)
#else
#define count_leading_zeroes_32(diff) ((diff >> 16) ? clz_table[diff >> 16] : 16 + clz_table[diff])
#endif

#if !defined(HAVE_SSE4_2)
#define count_trailing_zeroes_32(diff) mod_37_bit_position[(-diff & diff) % 37]
#elif defined(HAVE_TZCNT)
#define count_trailing_zeroes_32(diff) _tzcnt_u32(diff)
#elif defined(HAVE_BUILTIN_CTZ)
#define count_trailing_zeroes_32(diff) __builtin_ctz(diff)
#else
/* lowbit = -diff & diff */
#define count_trailing_zeroes_32(diff) mod_37_bit_position[(-diff & diff) % 37]
#endif


/* query is a substring of the original, starting with queryoffset */
/* Performs the same function as Sarray_lookup, which returns the diagonals */
static T
Elt_read_queryfwd (
#ifdef LARGE_GENOMES
		   unsigned char *positions_high,
#endif
		   UINT4 *positions, int n, int diagterm, int querylength, int querystart,
		   Compress_T query_compress, bool plusp, int genestrand) {
  T new;
  int max_nmatches;
  Univcoord_T *best_univdiagonals, *out, univdiagonal, left;
  int *nmatches, pos3;
  int i;
  

  debug(printf("Got %d positions\n",n));
  if (n == 0) {
    return (T) NULL;
  } else if (querystart >= querylength - index1part) {
    return (T) NULL;
  } else {
    max_nmatches = 0;
    nmatches = (int *) MALLOC(n*sizeof(int));
    for (i = 0; i < n; i++) {
#ifdef LARGE_GENOMES
      univdiagonal = GETPOS(positions_high[i],positions[i]) + diagterm;
      debug(printf("plusp %d, univdiagonal is %llu, querystart is %d\n",plusp,univdiagonal,querystart));
#else
      univdiagonal = positions[i] + diagterm;
      debug(printf("plusp %d, univdiagonal is %u, querystart is %d\n",plusp,univdiagonal,querystart));
#endif

      left = univdiagonal - (Univcoord_T) querylength;

      /* TRIM QUERY AT GENOME BOUNDS */
      pos3 = (univdiagonal <= genomelength) ? querylength : (int) (genomelength - left);

      /* Uses Genome_consecutive_matches_rightward, which is conservative */
      nmatches[i] = index1part +
	Genome_consecutive_matches_rightward(genomebits,query_compress,left,
					     /*pos5*/querystart+index1part,pos3,
					     plusp,genestrand);
      if (nmatches[i] > max_nmatches) {
	max_nmatches = nmatches[i];
      }
    }

    out = best_univdiagonals = MALLOC(n*sizeof(Univcoord_T));
    for (i = 0; i < n; i++) {
      if (nmatches[i] >= max_nmatches - index1interval + 1) {
#ifdef LARGE_GENOMES
	*out++ = GETPOS(positions_high[i],positions[i]) + diagterm;
#else
	*out++ = positions[i] + diagterm;
#endif
      }
    }
    FREE(nmatches);
   

    new = (T) MALLOC(sizeof(*new));
    new->qstart = querystart;
    new->qend = querystart + max_nmatches;
    new->nmatches = max_nmatches;

    assert(new->qend <= (int) querylength);
    debug(printf("Making an Elt with querystart %d, max_nmatches %d => max_queryend %d\n",
		 new->qstart,new->nmatches,new->qend));

    new->all_univdiagonals = new->univdiagonals = best_univdiagonals;
    new->n_all_univdiagonals = new->nunivdiagonals = out - best_univdiagonals;

#ifdef DEBUG
    printf("Univdiagonals:");
    for (i = 0; i < new->nunivdiagonals; i++) {
      printf(" %llu",new->univdiagonals[i]);
    }
    printf("\n");
#endif

    new->lowi = 0;
    new->highi = new->n_all_univdiagonals;

    return new;
  }
}


static T
Elt_read_queryrev (
#ifdef LARGE_GENOMES
		   unsigned char *positions_high,
#endif
		   UINT4 *positions, int n, int diagterm, int queryend, int querylength,
		   Compress_T query_compress, bool plusp, int genestrand) {
  T new;
  int max_nmatches;
  Univcoord_T *best_univdiagonals, *out, univdiagonal, left;
  int *nmatches, pos5;
  int i;
  

  debug(printf("Got %d positions\n",n));
  if (n == 0) {
    return (T) NULL;
  } else if (queryend < index1part) {
    return (T) NULL;
  } else {
    max_nmatches = 0;
    nmatches = (int *) MALLOC(n*sizeof(int));
    for (i = 0; i < n; i++) {
#ifdef LARGE_GENOMES
      univdiagonal = GETPOS(positions_high[i],positions[i]) + diagterm;
      debug(printf("plusp %d, univdiagonal is %llu, queryend is %d\n",plusp,univdiagonal,queryend));
#else
      univdiagonal = positions[i] + diagterm;
      debug(printf("plusp %d, univdiagonal is %u, queryend is %d\n",plusp,univdiagonal,queryend));
#endif

      left = univdiagonal - (Univcoord_T) querylength;

      /* TRIM QUERY AT GENOME BOUNDS */
      pos5 = (univdiagonal >= 0 + (Univcoord_T) querylength) ? 0 : (int) (0 - left);
      
      /* Uses Genome_consecutive_matches_rightward, which is conservative */
      nmatches[i] = index1part +
	Genome_consecutive_matches_leftward(genomebits,query_compress,left,
					    pos5,/*pos3*/queryend - index1part,plusp,genestrand);
      if (nmatches[i] > max_nmatches) {
	max_nmatches = nmatches[i];
      }
    }

    out = best_univdiagonals = MALLOC(n*sizeof(Univcoord_T));
    for (i = 0; i < n; i++) {
      if (nmatches[i] >= max_nmatches - index1interval + 1) {
#ifdef LARGE_GENOMES
	*out++ = GETPOS(positions_high[i],positions[i]) + diagterm;
#else
	*out++ = positions[i] + diagterm;
#endif
      }
    }
    FREE(nmatches);
   

    new = (T) MALLOC(sizeof(*new));
    new->qend = queryend;
    new->qstart = queryend - max_nmatches;
    new->nmatches = max_nmatches;

    assert(new->qstart >= 0);
    debug(printf("Making an Elt with queryend %d, max_nmatches %d => min_querystart %d\n",
		 new->qend,new->nmatches,new->qstart));

    new->all_univdiagonals = new->univdiagonals = best_univdiagonals;
    new->n_all_univdiagonals = new->nunivdiagonals = out - best_univdiagonals;

#ifdef DEBUG
    printf("Univdiagonals:");
    for (i = 0; i < new->nunivdiagonals; i++) {
      printf(" %llu",new->univdiagonals[i]);
    }
    printf("\n");
#endif

    new->lowi = 0;
    new->highi = new->n_all_univdiagonals;

    return new;
  }
}


#if 0
void
Elt_set_queryfwd_extend_left (List_T list, Compress_T query_compress,
			      int querylength, bool plusp, int genestrand) {
  List_T p;
  T this;
  int max_nmatches;
  Univcoord_T *best_univdiagonals, *out, univdiagonal, left;
  int *nmatches, pos5;
  int n, i;
  

  for (p = list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    n = this->n_all_univdiagonals;
    debug(printf("Starting with %d positions\n",n));
    assert(n > 0);

    max_nmatches = 0;
    nmatches = (int *) MALLOC(n*sizeof(int));
    for (i = 0; i < n; i++) {
      univdiagonal = this->all_univdiagonals[i];
      
      left = univdiagonal - (Univcoord_T) querylength;

      /* TRIM QUERY AT GENOME BOUNDS */
      pos5 = (univdiagonal >= 0 + (Univcoord_T) querylength) ? 0 : (int) (0 - left);

      /* Uses Genome_consecutive_matches_rightward, which is conservative */
      nmatches[i] = index1part +
	Genome_consecutive_matches_leftward(genomebits,query_compress,left,
					    pos5,/*pos3*/this->qend - index1part,plusp,genestrand);
      if (nmatches[i] > max_nmatches) {
	max_nmatches = nmatches[i];
      }
    }
    
    out = best_univdiagonals = MALLOC(n*sizeof(Univcoord_T));
    for (i = 0; i < n; i++) {
      if (nmatches[i] >= max_nmatches - index1interval + 1) {
	*out++ = this->all_univdiagonals[i];
      }
    }
    FREE(nmatches);
    
    this->qstart = this->qend - max_nmatches;
    this->nmatches = max_nmatches;
    
    debug(printf("Revising an Elt with queryend %d, nmatches %d => querystart %d\n",
		 this->qend,this->nmatches,this->qstart));
    FREE(this->all_univdiagonals);
    this->all_univdiagonals = this->univdiagonals = best_univdiagonals;
    this->n_all_univdiagonals = this->nunivdiagonals = out - best_univdiagonals;
    
#ifdef DEBUG
    printf("Univdiagonals:");
    for (i = 0; i < this->nunivdiagonals; i++) {
      printf(" %llu",this->univdiagonals[i]);
    }
    printf("\n");
#endif
    
    this->lowi = 0;
    this->highi = this->n_all_univdiagonals;
  }

  return;
}
#endif


#if 0
void
Elt_set_queryrev_extend_right (List_T list, int querylength,
			       Compress_T query_compress, bool plusp, int genestrand) {
  List_T p;
  T this;
  int max_nmatches;
  Univcoord_T *best_univdiagonals, *out, univdiagonal, left;
  int *nmatches;
  int n, i;
#ifdef LIBERAL
  int queryend;
  int nmismatches3;
#endif
  

  for (p = list; p != NULL; p = List_next(p)) {
    this = (T) List_head(p);
    n = this->n_all_univdiagonals;
    debug(printf("Starting with %d positions\n",n));
    assert(n > 0);

    max_nmatches = 0;
    nmatches = (int *) MALLOC(n*sizeof(int));
    for (i = 0; i < n; i++) {
      univdiagonal = this->all_univdiagonals[i];
      
      left = univdiagonal - (Univcoord_T) querylength;

      /* TRIM QUERY AT GENOME BOUNDS */
      pos3 = (univdiagonal <= genomelength) ? querylength : (int) (genomelength - left);

      /* Uses Genome_consecutive_matches_rightward, which is conservative */
      nmatches[i] = index1part +
	Genome_consecutive_matches_rightward(genomebits,query_compress,left,
					     /*pos5*/this->qstart + index1part,pos3,
					     plusp,genestrand);
      if (nmatches[i] > max_nmatches) {
	max_nmatches = nmatches[i];
      }
    }
    
    out = best_univdiagonals = MALLOC(n*sizeof(Univcoord_T));
    for (i = 0; i < n; i++) {
      if (nmatches[i] >= max_nmatches - index1interval + 1) {
	*out++ = this->all_univdiagonals[i];
      }
    }
    FREE(nmatches);
    
    
    this->qend = this->qstart + max_nmatches;
    this->nmatches = max_nmatches;
    
    debug(printf("Revising an Elt with querystart %d, nmatches %d => queryend %d\n",
		 this->qstart,this->nmatches,this->qend));
    FREE(this->all_univdiagonals);
    this->all_univdiagonals = this->univdiagonals = best_univdiagonals;
    this->n_all_univdiagonals = this->nunivdiagonals = out - best_univdiagonals;
    
#ifdef DEBUG
    printf("Univdiagonals:");
    for (i = 0; i < this->nunivdiagonals; i++) {
      printf(" %llu",this->univdiagonals[i]);
    }
    printf("\n");
#endif
    
    this->lowi = 0;
    this->highi = this->n_all_univdiagonals;
  }

  return;
}
#endif



static int
binary_search_univcoord (int lowi, int highi, Univcoord_T *univdiagonals, Univcoord_T goal) {
  int middlei;

  debug10(printf("entered binary search with lowi=%d, highi=%d, goal=%u\n",lowi,highi,goal));

  while (lowi < highi) {
    middlei = lowi + ((highi - lowi) / 2);
    debug10(printf("  binary: %d:%u %d:%u %d:%u   vs. %u\n",
		   lowi,univdiagonals[lowi],middlei,univdiagonals[middlei],
		   highi-1,univdiagonals[highi-1],goal));
    if (goal < univdiagonals[middlei]) {
      highi = middlei;
    } else if (goal > univdiagonals[middlei]) {
      lowi = middlei + 1;
    } else {
      debug10(printf("binary search returns %d\n",middlei));
      return middlei;
    }
  }

  debug10(printf("binary search returns %d\n",highi));
  return highi;
}


static void
Elt_filter_univdiagonals (T this, Univcoord_T low, Univcoord_T high) {
  int lowi, highi;
#ifdef DEBUG
  int i;
#endif

  debug(printf("Entered Elt_filter_univdiagonals on %d..%d with %d univdiagonals with low %u and high %u, nmatches %d\n",
	       this->qstart,this->qend,this->n_all_univdiagonals,low,high,this->nmatches));

  /* low_adj and high_adj are inclusive */
  lowi = binary_search_univcoord(/*lowi*/0,/*highi*/this->n_all_univdiagonals,this->all_univdiagonals,/*goal*/low);
  highi = binary_search_univcoord(lowi,/*highi*/this->n_all_univdiagonals,this->all_univdiagonals,/*goal*/high + 1) - 1;
  if ((this->nunivdiagonals = highi - lowi + 1) == 0) {
    this->univdiagonals = (Univcoord_T *) NULL;

  } else {
    this->univdiagonals = &(this->all_univdiagonals[lowi]);
  }

#ifdef DEBUG
  printf("Setting lowi %d and highi %d\n",lowi,highi);
  for (i = lowi; i <= highi; i++) {
    printf("  %u\n",this->all_univdiagonals[i]);
  }
#endif

  return;
}


#ifdef DEBUG
static void
Elt_dump (T elt) {
  int k;

  printf("Elt with max querybounds %d..%d and %d univdiagonals:\n",
	 elt->qstart,elt->qend,elt->nunivdiagonals);
  for (k = 0; k < elt->nunivdiagonals; k++) {
    printf("  %u\n",elt->univdiagonals[k]);
  }
  printf("\n");

  return;
}

static void
Elt_dump_set (List_T set) {
  List_T p;

  for (p = set; p != NULL; p = List_next(p)) {
    Elt_dump((T) List_head(p));
  }

  return;
}
#endif


#define add_bounded(x,plusterm,highbound) ((x + (plusterm) >= highbound) ? (highbound - 1) : x + (plusterm))
#define subtract_bounded(x,minusterm,lowbound) ((x < lowbound + (minusterm)) ? lowbound : x - (minusterm))


static bool
elt_startp (T elt, int middle_qstart, int middle_qend) {
  if (elt->qstart >= middle_qstart && elt->qend <= middle_qend) {
    debug13(printf("Not allowing left elt that is subsumed by middle elt: q %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else if (elt->qend >= middle_qend) {
    debug13(printf("Not allowing left elt that extends right of middle elt: q %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else if ((elt->qend - middle_qstart) > (middle_qend - middle_qstart) / 2) {
    debug13(printf("Not allowing left elt that mainly overlaps middle elt: q %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else if ((elt->qend - middle_qstart) > (elt->qend - elt->qstart) / 2) {
    debug13(printf("Not allowing left elt that mainly overlaps middle elt: q %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else {
    return true;
  }
}


static bool
elt_endp (T elt, int middle_qstart, int middle_qend) {
  if (elt->qstart >= middle_qstart && elt->qend <= middle_qend) {
    debug13(printf("Not allowing right elt that is subsumed by middle elt: qpos %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else if (elt->qstart <= middle_qstart) {
    debug13(printf("Not allowing right elt that extends left of middle elt: qpos %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else if ((middle_qend - elt->qstart) > (middle_qend - middle_qstart) / 2) {
    debug13(printf("Not allowing right elt that mainly overlaps middle elt: q %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else if ((middle_qend - elt->qstart) > (elt->qend - elt->qstart) / 2) {
    debug13(printf("Not allowing right elt that mainly overlaps middle elt: q %d..%d\n",
		   elt->qstart,elt->qend));
    return false;
  } else {
    return true;
  }
}


#if 0
static void
filter_elts (T *elt_array, int best_i, int nelts) {
  T elt, middle;
  int i;

  middle = elt_array[best_i];

  debug13(printf("Best elt is %d..%d with %d univdiagonals\n",middle->qstart,middle->qend,middle->nunivdiagonals));

  /* Filter elts and univdiagonals for right side */
  debug13(printf("Filtering elts on right side\n"));
  for (i = best_i + 1; i < nelts; i++) {
    elt = elt_array[i];
    debug13(printf("Handling right elt %d: %p\n",i,elt));
    if (elt->qstart >= middle->qstart && elt->qend <= middle->qend) {
      debug13(printf("Not allowing right elt that is subsumed by middle elt: qpos %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    } else if (elt->qstart <= middle->qstart) {
      debug13(printf("Not allowing right elt that extends left of middle elt: qpos %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    } else if ((middle->qend - elt->qstart) > (middle->qend - middle->qstart) / 2) {
      debug13(printf("Not allowing right elt that mainly overlaps middle elt: q %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    } else if ((middle->qend - elt->qstart) > (elt->qend - elt->qstart) / 2) {
      debug13(printf("Not allowing right elt that mainly overlaps middle elt: q %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    }
  }

  /* Filter elts and univdiagonals for left side */
  debug13(printf("Filtering elts on left side\n"));
  for (i = best_i - 1; i >= 0; --i) {
    elt = elt_array[i];
    debug13(printf("Handling left elt %d: %p\n",i,elt));
    if (elt->qstart >= middle->qstart && elt->qend <= middle->qend) {
      debug13(printf("Not allowing left elt that is subsumed by middle elt: q %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    } else if (elt->qend >= middle->qend) {
      debug13(printf("Not allowing left elt that extends right of middle elt: q %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    } else if ((elt->qend - middle->qstart) > (middle->qend - middle->qstart) / 2) {
      debug13(printf("Not allowing left elt that mainly overlaps middle elt: q %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    } else if ((elt->qend - middle->qstart) > (elt->qend - elt->qstart) / 2) {
      debug13(printf("Not allowing left elt that mainly overlaps middle elt: q %d..%d\n",
		     elt->qstart,elt->qend));
      /* Elt_free(&elt); */
      elt_array[i] = (Elt_T) NULL;
    }
  }

  return;
}
#endif



#if 0
static void
univdiagonals_dump (Univcoord_T *univdiagonals, int nunivdiagonals) {
  int i;

  for (i = 0; i < nunivdiagonals; i++) {
    printf(" %u",univdiagonals[i]);
  }
  printf("\n");

  return;
}
#endif


/* Caller needs to call Elt_gc(&plus_set) and Elt_gc(&minus_set) */
static void
get_elt_sets_queryfwd (List_T *plus_set, List_T *minus_set, T *best_plus_elt, T *best_minus_elt,
		       Stage1_T stage1, int querylength,
		       Compress_T query_compress_fwd, Compress_T query_compress_rev, 
		       int nmismatches_allowed, int genestrand, Listpool_T listpool) {
  T elt;
  int best_plus_nmatches, best_minus_nmatches;
  int max_plus_qpos, max_minus_qpos, plus_qpos[MAX_INDEX1INTERVAL], minus_qpos[MAX_INDEX1INTERVAL];

  int mod;
  int niter_plus, niter_minus;

#ifdef LARGE_GENOMES
  unsigned char *positions_high;
#endif
  UINT4 *positions;
  int npositions;

  int query_lastpos = querylength - index1part;
  int queryoffset;


  debug(printf("\nStarting get_elt_sets_queryfwd with querylength %d and nmismatches_allowed %d, genestrand %d\n",
	       querylength,nmismatches_allowed,genestrand));

#if 0
  /* Allow calls to Extension_search_queryfwd in addition to other methods */
  *hits_gplus = *hits_gminus = (List_T) NULL;
#endif

  if (nmismatches_allowed < 0) {
    nmismatches_allowed = 0;
#if 0
  } else {
    /* It is possible that this makes GSNAP too slow */
    nmismatches_allowed = querylength;
#endif
  }

  /* I.  Race from plus and minus start to end */
  best_plus_nmatches = best_minus_nmatches = 0;
  *best_plus_elt = *best_minus_elt = (T) NULL;
  max_plus_qpos = max_minus_qpos = 0;
  plus_qpos[0] = minus_qpos[0] = 0;
  plus_qpos[1] = minus_qpos[1] = 1;
  plus_qpos[2] = minus_qpos[2] = 2;
  niter_plus = niter_minus = 0;

  while (niter_plus <= nmismatches_allowed && niter_minus <= nmismatches_allowed &&
	 max_plus_qpos < query_lastpos && max_minus_qpos < query_lastpos) {
    for (mod = 0; mod < index1interval; mod++) {
      if ((queryoffset = plus_qpos[mod]) >= query_lastpos) {
	/* Skip */
	plus_qpos[mod] += index1interval;
      } else if (stage1->plus_validp[queryoffset] == false) {
	/* Skip */
	plus_qpos[mod] += index1interval;
      } else {
	if (stage1->plus_retrievedp[queryoffset] == true) {
#ifdef LARGE_GENOMES
	  positions_high = stage1->plus_positions_high[queryoffset];
#endif
	  positions = stage1->plus_positions[queryoffset];
	  npositions = stage1->plus_npositions[queryoffset];
	  debug(printf("Already have %d plus positions at %d\n",npositions,queryoffset));
	} else {
	  assert(stage1->plus_positions[queryoffset] == NULL);
	  stage1->plus_retrievedp[queryoffset] = true;
#ifdef LARGE_GENOMES
	  npositions = stage1->plus_npositions[queryoffset] =
	    Indexdb_largeptr(&stage1->plus_positions_high[queryoffset],&stage1->plus_positions[queryoffset],
			     indexdb,stage1->plus_oligos[queryoffset]);
	  positions_high = stage1->plus_positions_high[queryoffset];
#else
	  npositions = stage1->plus_npositions[queryoffset] =
	    Indexdb_ptr(&stage1->plus_positions[queryoffset],indexdb,stage1->plus_oligos[queryoffset]);
#endif
	  positions = stage1->plus_positions[queryoffset];
	}

	if ((elt = Elt_read_queryfwd(
#ifdef LARGE_GENOMES
				     positions_high,
#endif
				     positions,npositions,/*diagterm*/querylength - queryoffset,
				     querylength,/*querystart*/queryoffset,
				     query_compress_fwd,/*plusp*/true,genestrand)) == NULL) {
	  plus_qpos[mod] += index1interval;
	} else {
	  if (elt->nmatches > best_plus_nmatches && elt->n_all_univdiagonals <= MAX_HITS_FOR_BEST_ELT) {
	    *best_plus_elt = elt;
	    best_plus_nmatches = elt->nmatches;
	  }
	  *plus_set = Listpool_push(*plus_set,listpool,(void *) elt);
	  plus_qpos[mod] += elt->nmatches;
	  niter_plus++;
	}
      }

      if (plus_qpos[mod] > max_plus_qpos) {
	max_plus_qpos = plus_qpos[mod];
      }


      /* querypos_rc uses standard Stage1_T convention, but we have switched to sarray convention */
      /* querypos_rc = query_lastpos - queryoffset; */
      if ((queryoffset = minus_qpos[mod]) >= query_lastpos) {
	/* Skip */
	minus_qpos[mod] += index1interval;
      } else if (stage1->minus_validp[queryoffset] == false) {
	/* Skip */
	minus_qpos[mod] += index1interval;
      } else {
	if (stage1->minus_retrievedp[queryoffset] == true) {
#ifdef LARGE_GENOMES
	  positions_high = stage1->minus_positions_high[queryoffset];
#endif
	  positions = stage1->minus_positions[queryoffset];
	  npositions = stage1->minus_npositions[queryoffset];
	  debug(printf("Already have %d minus positions at %d\n",npositions,queryoffset));
	} else {
	  assert(stage1->minus_positions[queryoffset] == NULL);
	  stage1->minus_retrievedp[queryoffset] = true;
	  /* In standard stage1 convention, diagterm would be
	     -querylength + querypos + index1part = -(query_lastpos - querypos) = -queryoffset */
#ifdef LARGE_GENOMES
	  npositions = stage1->minus_npositions[queryoffset] =
	    Indexdb_largeptr(&stage1->minus_positions_high[queryoffset],&stage1->minus_positions[queryoffset],
			     indexdb2,stage1->minus_oligos[queryoffset]);
	  positions_high = stage1->minus_positions_high[queryoffset];
#else
	  npositions = stage1->minus_npositions[queryoffset] =
	    Indexdb_ptr(&stage1->minus_positions[queryoffset],indexdb2,stage1->minus_oligos[queryoffset]);
#endif
	  positions = stage1->minus_positions[queryoffset];
	}
	
	if ((elt = Elt_read_queryfwd(
#ifdef LARGE_GENOMES
				     positions_high,
#endif
				     positions,npositions,/*diagterm*/querylength - queryoffset,
				     querylength,/*querystart*/queryoffset,
				     query_compress_rev,/*plusp*/false,genestrand)) == NULL) {
	  minus_qpos[mod] += index1interval;
	} else {
	  if (elt->nmatches > best_minus_nmatches && elt->n_all_univdiagonals <= MAX_HITS_FOR_BEST_ELT) {
	    *best_minus_elt = elt;
	    best_minus_nmatches = elt->nmatches;
	  }
	  *minus_set = Listpool_push(*minus_set,listpool,(void *) elt);
	  minus_qpos[mod] += elt->nmatches;
	  niter_minus++;
	}
      }
	
      if (minus_qpos[mod] > max_minus_qpos) {
	max_minus_qpos = minus_qpos[mod];
      }
    }

#ifdef DEBUG
    printf("\n");
    for (mod = 0; mod < index1interval; mod++) {
      printf("mod %d, plus_qpos %d\n",mod,plus_qpos[mod]);
      printf("mod %d, minus_qpos %d\n",mod,minus_qpos[mod]);
    }
    printf("max_plus_qpos %d\n",max_plus_qpos);
    printf("max_maxus_qpos %d\n",max_minus_qpos);
    printf("\n");
#endif

#ifndef LIBERAL
    /* Skip the presumed mismatch */
    max_plus_qpos += 1;
    max_minus_qpos += 1;
#endif

    plus_qpos[0] = max_plus_qpos;
    plus_qpos[1] = max_plus_qpos + 1;
    plus_qpos[2] = max_plus_qpos + 2;

    minus_qpos[0] = max_minus_qpos;
    minus_qpos[1] = max_minus_qpos + 1;
    minus_qpos[2] = max_minus_qpos + 2;
  }

  *plus_set = List_reverse(*plus_set);
  *minus_set = List_reverse(*minus_set);

#ifdef DEBUG
  printf("queryfwd plus set:\n");
  Elt_dump_set(*plus_set);
  printf("\n");
  printf("queryfwd minus set:\n");
  Elt_dump_set(*minus_set);
  printf("\n");
#endif

  if (max_minus_qpos >= query_lastpos) {
    /* This branch is critical for high speed (about 4x faster),
       although we could miss some hits that need to be solved by
       another method */
    debug(printf("QUERYFWD PLUS: minus side won, so skip plus side\n"));
    *best_plus_elt = (Elt_T) NULL;

  } else if (max_plus_qpos < query_lastpos || *best_plus_elt == NULL) {
    debug(printf("QUERYFWD PLUS: Still could not find large pieces: plus_qpos %d < query_lastpos %d, best_plus_elt %p\n",
		 max_plus_qpos,query_lastpos,*best_plus_elt));
    *best_plus_elt = (Elt_T) NULL;

  } else if ((*best_plus_elt)->nunivdiagonals == 0) {
      /* Could happen if there are too many univdiagonals */
    debug(printf("QUERYFWD PLUS: Best elt has no univdiagonals\n"));
    *best_plus_elt = (Elt_T) NULL;

  } else {
    debug(printf("QUERYFWD PLUS HAS BEST ELT: "));
    debug(Elt_dump(*best_plus_elt));
    debug(printf("\n"));
  }

  if (max_plus_qpos >= query_lastpos) {
    /* This branch is critical for high speed (about 4x faster),
       although we could miss some hits that need to be solved by
       another method */
    debug(printf("QUERYFWD MINUS: plus side won, so skip minus side\n"));
    *best_minus_elt = (Elt_T) NULL;

  } else if (max_minus_qpos < query_lastpos || *best_minus_elt == NULL) {
    debug(printf("QUERYFWD MINUS: Still could not find large pieces: minus_qpos %d < query_lastpos %d, best_minus_elt %p\n",
		 max_minus_qpos,query_lastpos,*best_minus_elt));
    *best_minus_elt = (Elt_T) NULL;

  } else if ((*best_minus_elt)->nunivdiagonals == 0) {
    /* Could happen if there are too many univdiagonals */
    debug(printf("QUERYFWD MINUS: Best elt has no univdiagonals\n"));
    *best_minus_elt = (Elt_T) NULL;

  } else {
    debug(printf("QUERYFWD MINUS HAS BEST ELT: "));
    debug(Elt_dump(*best_minus_elt));
    debug(printf("\n"));
  }

  return;
}


/* Note: A second try, starting at queryrev, may solve only a few more
   cases presented to this method (i.e., cases that cannot be solved
   by kmer-end search).  It therefore may be best to push these cases
   to the next method. */

/* Caller needs to call Elt_gc(&plus_set) and Elt_gc(&minus_set) */
static void
get_elt_sets_queryrev (List_T *plus_set, List_T *minus_set, T *best_plus_elt, T *best_minus_elt,
		       Stage1_T stage1, int querylength,
		       Compress_T query_compress_fwd, Compress_T query_compress_rev, 
		       int nmismatches_allowed, int genestrand, Listpool_T listpool) {
  T elt;
  int best_plus_nmatches, best_minus_nmatches;
  int min_plus_qpos, min_minus_qpos, plus_qpos[MAX_INDEX1INTERVAL], minus_qpos[MAX_INDEX1INTERVAL];

  int mod;
  int niter_plus, niter_minus;

#ifdef LARGE_GENOMES
  unsigned char *positions_high;
#endif
  UINT4 *positions;
  int npositions;

  int query_lastpos = querylength - index1part;
  int queryoffset;

  debug(printf("\nStarting get_elt_sets_queryrev with querylength %d and nmismatches_allowed %d, genestrand %d\n",
	       querylength,nmismatches_allowed,genestrand));

#if 0
  /* Allow calls to Extension_search_queryrev in addition to other methods */
  *hits_gplus = *hits_gminus = (List_T) NULL;
#endif

  if (nmismatches_allowed < 0) {
    nmismatches_allowed = 0;
#if 0
  } else {
    /* It is possible that this makes GSNAP too slow */
    nmismatches_allowed = querylength;
#endif
  }

  /* I.  Race from plus and minus end to start */
  best_plus_nmatches = best_minus_nmatches = 0;
  *best_plus_elt = *best_minus_elt = (T) NULL;
  min_plus_qpos = min_minus_qpos = query_lastpos;
  plus_qpos[0] = minus_qpos[0] = query_lastpos;
  plus_qpos[1] = minus_qpos[1] = query_lastpos-1;
  plus_qpos[2] = minus_qpos[2] = query_lastpos-2;
  niter_plus = niter_minus = 0;

  while (niter_plus <= nmismatches_allowed && niter_minus <= nmismatches_allowed &&
	 min_plus_qpos > 0 && min_minus_qpos > 0) {
    for (mod = 0; mod < index1interval; mod++) {
      if ((queryoffset = plus_qpos[mod]) < 0) {
	/* Skip */
	plus_qpos[mod] -= index1interval;
      } else if (stage1->plus_validp[queryoffset] == false) {
	/* Skip */
	plus_qpos[mod] -= index1interval;
      } else {
	if (stage1->plus_retrievedp[queryoffset] == true) {
#ifdef LARGE_GENOMES
	  positions_high = stage1->plus_positions_high[queryoffset];
#endif
	  positions = stage1->plus_positions[queryoffset];
	  npositions = stage1->plus_npositions[queryoffset];
	} else {
	  assert(stage1->plus_positions[queryoffset] == NULL);
	  stage1->plus_retrievedp[queryoffset] = true;
#ifdef LARGE_GENOMES
	  npositions = stage1->plus_npositions[queryoffset] =
	    Indexdb_largeptr(&stage1->plus_positions_high[queryoffset],&stage1->plus_positions[queryoffset],
			     indexdb,stage1->plus_oligos[queryoffset]);
	  positions_high = stage1->plus_positions_high[queryoffset];
#else
	  npositions = stage1->plus_npositions[queryoffset] =
	    Indexdb_ptr(&stage1->plus_positions[queryoffset],indexdb,stage1->plus_oligos[queryoffset]);
#endif
	  positions = stage1->plus_positions[queryoffset];
	}
	
	if ((elt = Elt_read_queryrev(
#ifdef LARGE_GENOMES
				     positions_high,
#endif
				     positions,npositions,/*diagterm*/querylength - queryoffset,
				     /*queryend*/queryoffset + index1part,querylength,
				     query_compress_fwd,/*plusp*/true,genestrand)) == NULL) {
	  plus_qpos[mod] -= index1interval;
	} else {
	  if (elt->nmatches > best_plus_nmatches && elt->n_all_univdiagonals <= MAX_HITS_FOR_BEST_ELT) {
	    *best_plus_elt = elt;
	    best_plus_nmatches = elt->nmatches;
	  }
	  *plus_set = Listpool_push(*plus_set,listpool,(void *) elt);
	  plus_qpos[mod] -= elt->nmatches;
	  niter_plus++;
	}
      }

      if (plus_qpos[mod] < min_plus_qpos) {
	min_plus_qpos = plus_qpos[mod];
      }


      /* querypos_rc uses standard Stage1_T convention, but we have switched to sarray convention */
      /* querypos_rc = query_lastpos - queryoffset; */
      if ((queryoffset = minus_qpos[mod]) < 0) {
	/* Skip */
	minus_qpos[mod] -= index1interval;
      } else if (stage1->minus_validp[queryoffset] == false) {
	/* Skip */
	minus_qpos[mod] -= index1interval;
      } else {
	if (stage1->minus_retrievedp[queryoffset] == true) {
#ifdef LARGE_GENOMES
	  positions_high = stage1->minus_positions_high[queryoffset];
#endif
	  positions = stage1->minus_positions[queryoffset];
	  npositions = stage1->minus_npositions[queryoffset];
	} else {
	  assert(stage1->minus_positions[queryoffset] == NULL);
	  stage1->minus_retrievedp[queryoffset] = true;
	  /* In standard stage1 convention, diagterm would be
	     -querylength + querypos + index1part = -(query_lastpos - querypos) = -queryoffset */
#ifdef LARGE_GENOMES
	  npositions = stage1->minus_npositions[queryoffset] =
	    Indexdb_largeptr(&stage1->minus_positions_high[queryoffset],&stage1->minus_positions[queryoffset],
			     indexdb2,stage1->minus_oligos[queryoffset]);
	  positions_high = stage1->minus_positions_high[queryoffset];
#else
	  npositions = stage1->minus_npositions[queryoffset] =
	    Indexdb_ptr(&stage1->minus_positions[queryoffset],indexdb2,stage1->minus_oligos[queryoffset]);
#endif
	  positions = stage1->minus_positions[queryoffset];
	}
	
	if ((elt = Elt_read_queryrev(
#ifdef LARGE_GENOMES
				     positions_high,
#endif
				     positions,npositions,/*diagterm*/querylength - queryoffset,
				     /*queryend*/queryoffset + index1part,querylength,
				     query_compress_rev,/*plusp*/false,genestrand)) == NULL) {
	  minus_qpos[mod] -= index1interval;
	} else {
	  if (elt->nmatches > best_minus_nmatches && elt->n_all_univdiagonals <= MAX_HITS_FOR_BEST_ELT) {
	    *best_minus_elt = elt;
	    best_minus_nmatches = elt->nmatches;
	  }
	  *minus_set = Listpool_push(*minus_set,listpool,(void *) elt);
	  minus_qpos[mod] -= elt->nmatches;
	  niter_minus++;
	}
      }

      if (minus_qpos[mod] < min_minus_qpos) {
	min_minus_qpos = minus_qpos[mod];
      }
    }

#ifdef DEBUG
    printf("\n");
    for (mod = 0; mod < index1interval; mod++) {
      printf("mod %d, plus_qpos %d\n",mod,plus_qpos[mod]);
      printf("mod %d, minus_qpos %d\n",mod,minus_qpos[mod]);
    }
    printf("min_plus_qpos %d\n",min_plus_qpos);
    printf("min_minus_qpos %d\n",min_minus_qpos);
    printf("\n");
#endif

#ifndef LIBERAL
    /* Skip the presumed mismatch */
    min_plus_qpos -= 1;
    min_minus_qpos -= 1;
#endif

    plus_qpos[0] = min_plus_qpos;
    plus_qpos[1] = min_plus_qpos - 1;
    plus_qpos[2] = min_plus_qpos - 2;

    minus_qpos[0] = min_minus_qpos;
    minus_qpos[1] = min_minus_qpos - 1;
    minus_qpos[2] = min_minus_qpos - 2;
  }

#if 0
  /* Not needed for queryrev */
  *plus_set = List_reverse(*plus_set);
  *minus_set = List_reverse(*minus_set);
#endif

#ifdef DEBUG
  printf("queryrev plus set:\n");
  Elt_dump_set(*plus_set);
  printf("\n");
  printf("queryrev minus set:\n");
  Elt_dump_set(*minus_set);
  printf("\n");
#endif

  if (min_minus_qpos <= 0) {
    /* This branch is critical for high speed (about 4x faster),
       although we could miss some hits that need to be solved by
       another method */
    debug(printf("QUERYREV PLUS: minus side won, so skip plus side\n"));
    *best_plus_elt = (Elt_T) NULL;

  } else if (min_plus_qpos > 0 || *best_plus_elt == NULL) {
    debug(printf("QUERYREV PLUS: Still could not find large pieces: plus_qpos %d > 0, best_plus_elt %p\n",
		 min_plus_qpos,*best_plus_elt));
    *best_plus_elt = (Elt_T) NULL;

  } else if ((*best_plus_elt)->nunivdiagonals == 0) {
    /* Could happen if there are too many univdiagonals */
    debug(printf("QUERYREV PLUS: Best elt has no univdiagonals\n"));
    *best_plus_elt = (Elt_T) NULL;

  } else {
    debug(printf("QUERYREV PLUS HAS BEST ELT: "));
    debug(Elt_dump(*best_plus_elt));
    debug(printf("\n"));
  }

  if (min_plus_qpos <= 0) {
    /* This branch is critical for high speed (about 4x faster),
       although we could miss some hits that need to be solved by
       another method */
    debug(printf("QUERYREV MINUS: plus side won, so skip minus side\n"));
    *best_minus_elt = (Elt_T) NULL;

  } else if (min_minus_qpos > 0 || *best_minus_elt == NULL) {
    debug(printf("QUERYREV MINUS: Still could not find large pieces: minus_qpos %d > 0, best_minus_elt %p\n",
		 min_minus_qpos,*best_minus_elt));
    *best_minus_elt = (Elt_T) NULL;

  } else if ((*best_minus_elt)->nunivdiagonals == 0) {
    /* Could happen if there are too many univdiagonals */
    debug(printf("QUERYREV MINUS: Best elt has no univdiagonals\n"));
    *best_minus_elt = (Elt_T) NULL;
    
  } else {
    debug(printf("QUERYREV MINUS HAS BEST ELT: "));
    debug(Elt_dump(*best_minus_elt));
    debug(printf("\n"));
  }

  return;
}


static void
process_seed (int *found_score_overall, int *found_score_within_trims,
	      Chrnum_T *chrnum, Univcoord_T *chroffset, Univcoord_T *chrhigh,
	      Chrpos_T *chrlength, List_T *sense_hits, List_T *antisense_hits,
	      Univcoord_T middle_univdiagonal, int middle_qstart, int middle_qend,
	      List_T queryfwd_set, List_T queryrev_set, T queryfwd_best_elt, T queryrev_best_elt,
	      Stage1_T stage1, char *queryptr, int querylength, int *mismatch_positions_alloc, Compress_T query_compress,
	      int nmismatches_allowed, int max_insertionlen, int max_deletionlen, Chrpos_T overall_end_distance,
	      bool plusp, int genestrand, bool paired_end_p, bool first_read_p,
	      Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool,
	      Listpool_T listpool, Univdiagpool_T univdiagpool, Hitlistpool_T hitlistpool,
	      int level) {

  List_T left_univdiagonals = NULL, right_univdiagonals = NULL, p;
  Univdiag_T *left_univdiagonal_array, *right_univdiagonal_array;
  Univcoord_T univdiagonal, left, start_low, start_high, end_low, end_high;
  int qstart, qend;
  Elt_T elt;
  int nleft, nright, i, j, k;
  bool foundp;


  /* middle_qstart and middle_qend are the maximum bounds for the entire elt, so for an individual univdiagonal,
     they might go past the chromosome boundaries */

  left = middle_univdiagonal - querylength; /* Can be negative because we add middle_qstart or middle_qend */

  /* TRIM QUERY AT GENOME BOUNDS */
  middle_qstart = (middle_univdiagonal + middle_qstart >= 0 + (Univcoord_T) querylength) ? middle_qstart : (int) (0 - left);
  middle_qend = (middle_univdiagonal + middle_qend <= genomelength + (Univcoord_T) querylength) ? middle_qend : (int) (genomelength - left);

  /* Need to update chrnum, chroffset, chrhigh, chrlength through multiple calls to process_seed */
  *chrhigh = Univ_IIT_update_chrnum(&(*chrnum),&(*chroffset),*chrhigh,&(*chrlength),chromosome_iit,
				    /*low*/left + middle_qstart,/*high*/left + middle_qend,circular_typeint);
  
  /* TRIM QUERY AT CHROMOSOME BOUNDS */
  middle_qstart = (middle_univdiagonal + middle_qstart >= (*chroffset) + (Univcoord_T) querylength) ? middle_qstart : (int) ((*chroffset) - left);
  middle_qend = (middle_univdiagonal + middle_qend <= (*chrhigh) + (Univcoord_T) querylength) ? middle_qend : (int) ((*chrhigh) - left);
  debug(printf("PROCESS SEED at chrnum #%d, %u [%u], qstart %d, qend %d\n",
	       *chrnum,middle_univdiagonal,middle_univdiagonal - (*chroffset),middle_qstart,middle_qend));


  start_low = subtract_bounded(middle_univdiagonal,/*minusterm*/overall_end_distance,(*chroffset) + querylength);
  start_high = add_bounded(middle_univdiagonal,/*plusterm*/max_insertionlen,*chrhigh);

  end_low = subtract_bounded(middle_univdiagonal,/*minusterm*/max_insertionlen,(*chroffset) + querylength);
  end_high = add_bounded(middle_univdiagonal,/*plusterm*/overall_end_distance,*chrhigh);

  debug(printf("Computing start %u..%u and end %u..%u.  Chromosome bounds: %u..%u\n",
	       start_low,start_high,end_low,end_high,*chroffset,*chrhigh));


  for (p = queryfwd_set; p != NULL; p = List_next(p)) {
    elt = (Elt_T) List_head(p);
    if (elt != queryfwd_best_elt && elt != queryrev_best_elt) {
      if (elt_startp(elt,middle_qstart,middle_qend) == true) {
	Elt_filter_univdiagonals(elt,start_low,start_high);
	for (k = 0; k < elt->nunivdiagonals; k++) {
	  left_univdiagonals = Univdiagpool_push(left_univdiagonals,univdiagpool,elt->qstart,elt->qend,elt->univdiagonals[k]);
	  debug(printf("Pushing %u onto left of query fwd\n",elt->univdiagonals[k]));
	}
      }
      
      if (elt_endp(elt,middle_qstart,middle_qend) == true) {
	Elt_filter_univdiagonals(elt,end_low,end_high);
	for (k = 0; k < elt->nunivdiagonals; k++) {
	  right_univdiagonals = Univdiagpool_push(right_univdiagonals,univdiagpool,elt->qstart,elt->qend,elt->univdiagonals[k]);
	  debug(printf("Pushing %u onto right of query fwd\n",elt->univdiagonals[k]));
	}
      }
    }
  }
  
  for (p = queryrev_set; p != NULL; p = List_next(p)) {
    elt = (Elt_T) List_head(p);
    if (elt != queryfwd_best_elt && elt != queryrev_best_elt) {
      if (elt_startp(elt,middle_qstart,middle_qend) == true) {
	Elt_filter_univdiagonals(elt,start_low,start_high);
	for (k = 0; k < elt->nunivdiagonals; k++) {
	  left_univdiagonals = Univdiagpool_push(left_univdiagonals,univdiagpool,elt->qstart,elt->qend,elt->univdiagonals[k]);
	  debug(printf("Pushing %u onto left of query rev\n",elt->univdiagonals[k]));
	}
      }
      
      if (elt_endp(elt,middle_qstart,middle_qend) == true) {
	Elt_filter_univdiagonals(elt,end_low,end_high);
	for (k = 0; k < elt->nunivdiagonals; k++) {
	  right_univdiagonals = Univdiagpool_push(right_univdiagonals,univdiagpool,elt->qstart,elt->qend,elt->univdiagonals[k]);
	  debug(printf("Pushing %u onto right of query rev\n",elt->univdiagonals[k]));
	}
      }
    }
  }

  if (left_univdiagonals != NULL) {
    /* Sort the left univdiagonals and unique them */
    left_univdiagonal_array = (Univdiag_T *) List_to_array_n(&nleft,left_univdiagonals);
    qsort(left_univdiagonal_array,nleft,sizeof(Univdiag_T),Univdiag_diagonal_cmp);
    /* List_free(&left_univdiagonals); -- allocated by Univdiagpool_push */

    debug(printf("left univdiagonals is not NULL => %d\n",nleft));

    left_univdiagonals = (List_T) NULL;
    i = 0;
    while (i < nleft) {
      univdiagonal = left_univdiagonal_array[i]->univdiagonal;
      qstart = left_univdiagonal_array[i]->qstart;
      qend = left_univdiagonal_array[i]->qend;
      
      j = i+1;
      while (j < nleft && left_univdiagonal_array[j]->univdiagonal == univdiagonal) {
	if (left_univdiagonal_array[j]->qstart < qstart) {
	  qstart = left_univdiagonal_array[j]->qstart;
	}
	if (left_univdiagonal_array[j]->qend > qend) {
	  qend = left_univdiagonal_array[j]->qend;
	}
	j++;
      }
      
      left_univdiagonal_array[i]->qstart = qstart;
      left_univdiagonal_array[i]->qend = qend;
      left_univdiagonals = Univdiagpool_push_existing(left_univdiagonals,univdiagpool,left_univdiagonal_array[i]);
      i = j;
    }
    FREE(left_univdiagonal_array);
  }

  if (right_univdiagonals != NULL) {
    /* Sort the right univdiagonals and unique them */
    right_univdiagonal_array = (Univdiag_T *) List_to_array_n(&nright,right_univdiagonals);
    qsort(right_univdiagonal_array,nright,sizeof(Univdiag_T),Univdiag_diagonal_cmp);
    /* List_free(&right_univdiagonals); -- allocated by Univdiagpool_push */
    
    debug(printf("right univdiagonals is not NULL => %d\n",nright));

    right_univdiagonals = (List_T) NULL;
    i = 0;
    while (i < nright) {
      univdiagonal = right_univdiagonal_array[i]->univdiagonal;
      qstart = right_univdiagonal_array[i]->qstart;
      qend = right_univdiagonal_array[i]->qend;
      
      j = i+1;
      while (j < nright && right_univdiagonal_array[j]->univdiagonal == univdiagonal) {
	if (right_univdiagonal_array[j]->qstart < qstart) {
	  qstart = right_univdiagonal_array[j]->qstart;
	}
	if (right_univdiagonal_array[j]->qend > qend) {
	  qend = right_univdiagonal_array[j]->qend;
	}
	j++;
      }
      
      right_univdiagonal_array[i]->qstart = qstart;
      right_univdiagonal_array[i]->qend = qend;
      right_univdiagonals = Univdiagpool_push_existing(right_univdiagonals,univdiagpool,right_univdiagonal_array[i]);
      i = j;
    }
    FREE(right_univdiagonal_array);
  }


  debug(printf("Calling Path_solve with %d left and %d right diagonals for %u [%u]\n",
	       List_length(left_univdiagonals),List_length(right_univdiagonals),middle_univdiagonal,
	       middle_univdiagonal - (*chroffset)));
  Path_solve_from_diagonals(&foundp,&(*found_score_overall),&(*found_score_within_trims),
			    &(*sense_hits),&(*antisense_hits),
			    middle_univdiagonal,middle_qstart,middle_qend,
			    right_univdiagonals,left_univdiagonals,queryptr,querylength,
			    mismatch_positions_alloc,stage1->spliceinfo,
			    stage1->stream_alloc,stage1->streamsize_alloc,query_compress,
			    *chrnum,*chroffset,*chrhigh,*chrlength,plusp,genestrand,
			    nmismatches_allowed,max_insertionlen,max_deletionlen,overall_end_distance,
			    paired_end_p,first_read_p,intlistpool,univcoordlistpool,listpool,univdiagpool,
			    hitlistpool,/*method*/EXT,level);

  /* List_free(&right_univdiagonals); -- allocated by Univdiagpool_push */
  /* List_free(&left_univdiagonals); -- allocated by Univdiagpool_push */

  return;
}


#define min(a,b) (a < b) ? a : b
#define max(a,b) (a > b) ? a : b


static void
extend_seeds (int *found_score_overall, int *found_score_within_trims,
	      List_T *sense_hits, List_T *antisense_hits,
	      T queryfwd_best_elt, T queryrev_best_elt,
	      List_T queryfwd_set, List_T queryrev_set,

	      Stage1_T stage1, char *queryptr, int querylength,
	      int *mismatch_positions_alloc, Compress_T query_compress,
	      int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
	      Chrpos_T overall_end_distance,
	      bool plusp, int genestrand, bool paired_end_p, bool first_read_p,
	      Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool,
	      Listpool_T listpool, Univdiagpool_T univdiagpool, Hitlistpool_T hitlistpool,
	      int level) {

  Univcoord_T queryfwd_univdiagonal, queryrev_univdiagonal;
  Chrnum_T chrnum;
  Univcoord_T chroffset, chrhigh, univdiagonal;
  Chrpos_T chrlength;
  int qstart, qend;
  int queryfwd_nseeds, queryrev_nseeds, i, j;


  chrnum = 1;
  Univ_IIT_interval_bounds(&chroffset,&chrhigh,&chrlength,chromosome_iit,/*chrnum*/1,circular_typeint);

  if (queryfwd_best_elt == NULL) {
    queryfwd_nseeds = 0;
  } else {
    queryfwd_nseeds = queryfwd_best_elt->n_all_univdiagonals;
  }

  if (queryrev_best_elt == NULL) {
    queryrev_nseeds = 0;
  } else {
    queryrev_nseeds = queryrev_best_elt->n_all_univdiagonals;
  }
    
  /* Go through the union of the seeds */
  i = j = 0;
  while (i < queryfwd_nseeds && j < queryrev_nseeds) {
    queryfwd_univdiagonal = queryfwd_best_elt->all_univdiagonals[i];
    queryrev_univdiagonal = queryrev_best_elt->all_univdiagonals[j];

    if (queryfwd_univdiagonal == queryrev_univdiagonal) {
      /* Combine the seeds */
      univdiagonal = queryfwd_univdiagonal;
      qstart = min(queryfwd_best_elt->qstart,queryrev_best_elt->qstart);
      qend = max(queryfwd_best_elt->qend,queryrev_best_elt->qend);
      i++; j++;

    } else if (queryfwd_univdiagonal < queryrev_univdiagonal) {
      univdiagonal = queryfwd_univdiagonal;
      qstart = queryfwd_best_elt->qstart;
      qend = queryfwd_best_elt->qend;
      i++;

    } else {
      univdiagonal = queryrev_univdiagonal;
      qstart = queryrev_best_elt->qstart;
      qend = queryrev_best_elt->qend;
      j++;
    }

    /* Disallow univdiagonals from queryfwd_best_elt or queryrev_best_elt */
    process_seed(&(*found_score_overall),&(*found_score_within_trims),
		 &chrnum,&chroffset,&chrhigh,&chrlength,&(*sense_hits),&(*antisense_hits),
		 univdiagonal,qstart,qend,queryfwd_set,queryrev_set,queryfwd_best_elt,queryrev_best_elt,
		 stage1,queryptr,querylength,mismatch_positions_alloc,
		 query_compress,nmismatches_allowed,
		 max_insertionlen,max_deletionlen,overall_end_distance,
		 plusp,genestrand,paired_end_p,first_read_p,
		 intlistpool,univcoordlistpool,listpool,univdiagpool,hitlistpool,level);
  }

  if (i < queryfwd_nseeds) {
    qstart = queryfwd_best_elt->qstart;
    qend = queryfwd_best_elt->qend;
    while (i < queryfwd_nseeds) {
      univdiagonal = queryfwd_best_elt->all_univdiagonals[i++];
      
      /* Allow univdiagonals from queryrev_best_elt */
      process_seed(&(*found_score_overall),&(*found_score_within_trims),
		   &chrnum,&chroffset,&chrhigh,&chrlength,&(*sense_hits),&(*antisense_hits),
		   univdiagonal,qstart,qend,queryfwd_set,queryrev_set,queryfwd_best_elt,/*queryrev_best_elt*/NULL,
		   stage1,queryptr,querylength,mismatch_positions_alloc,
		   query_compress,nmismatches_allowed,
		   max_insertionlen,max_deletionlen,overall_end_distance,
		   plusp,genestrand,paired_end_p,first_read_p,
		   intlistpool,univcoordlistpool,listpool,univdiagpool,hitlistpool,level);
    }
  }

  if (j < queryrev_nseeds) {
    qstart = queryrev_best_elt->qstart;
    qend = queryrev_best_elt->qend;
    while (j < queryrev_nseeds) {
      univdiagonal = queryrev_best_elt->all_univdiagonals[j++];
      
      /* Allow univdiagonals from queryfwd_best_elt */
      process_seed(&(*found_score_overall),&(*found_score_within_trims),
		   &chrnum,&chroffset,&chrhigh,&chrlength,&(*sense_hits),&(*antisense_hits),
		   univdiagonal,qstart,qend,queryfwd_set,queryrev_set,/*queryfwd_best_elt*/NULL,queryrev_best_elt,
		   stage1,queryptr,querylength,mismatch_positions_alloc,
		   query_compress,nmismatches_allowed,
		   max_insertionlen,max_deletionlen,overall_end_distance,
		   plusp,genestrand,paired_end_p,first_read_p,
		   intlistpool,univcoordlistpool,listpool,univdiagpool,hitlistpool,level);
    }
  }

  return;
}


void
Extension_search (int *found_score_overall, int *found_score_within_trims,
		  List_T *sense_hits_gplus, List_T *sense_hits_gminus,
		  List_T *antisense_hits_gplus, List_T *antisense_hits_gminus,
		  Stage1_T stage1, char *queryuc_ptr, char *queryrc, int querylength,
		  int *mismatch_positions_alloc, Compress_T query_compress_fwd, Compress_T query_compress_rev, 

		  Intlistpool_T intlistpool, Univcoordlistpool_T univcoordlistpool, Listpool_T listpool,
		  Univdiagpool_T univdiagpool, Hitlistpool_T hitlistpool,
		  int nmismatches_allowed, int max_insertionlen, int max_deletionlen,
		  Chrpos_T overall_end_distance,
		  int genestrand, bool paired_end_p, bool first_read_p, int level) {

  T queryfwd_best_plus_elt, queryrev_best_plus_elt, queryfwd_best_minus_elt, queryrev_best_minus_elt;


  get_elt_sets_queryfwd(&stage1->queryfwd_plus_set,&stage1->queryfwd_minus_set,
			&queryfwd_best_plus_elt,&queryfwd_best_minus_elt,
			stage1,querylength,query_compress_fwd,query_compress_rev,
			nmismatches_allowed,genestrand,listpool);
  
  get_elt_sets_queryrev(&stage1->queryrev_plus_set,&stage1->queryrev_minus_set,
			&queryrev_best_plus_elt,&queryrev_best_minus_elt,
			stage1,querylength,query_compress_fwd,query_compress_rev,
			nmismatches_allowed,genestrand,listpool);

  extend_seeds(&(*found_score_overall),&(*found_score_within_trims),
	       &(*sense_hits_gplus),&(*antisense_hits_gplus),
	       queryfwd_best_plus_elt,queryrev_best_plus_elt,
	       stage1->queryfwd_plus_set,stage1->queryrev_plus_set,
	       stage1,/*queryptr*/queryuc_ptr,querylength,mismatch_positions_alloc,
	       /*query_compress*/query_compress_fwd,nmismatches_allowed,
	       max_insertionlen,max_deletionlen,overall_end_distance,
	       /*plusp*/true,genestrand,paired_end_p,first_read_p,
	       intlistpool,univcoordlistpool,listpool,univdiagpool,
	       hitlistpool,level);

  extend_seeds(&(*found_score_overall),&(*found_score_within_trims),
	       &(*sense_hits_gminus),&(*antisense_hits_gminus),
	       queryfwd_best_minus_elt,queryrev_best_minus_elt,
	       stage1->queryfwd_minus_set,stage1->queryrev_minus_set,
	       stage1,/*queryptr*/queryrc,querylength,mismatch_positions_alloc,
	       /*query_compress*/query_compress_rev,nmismatches_allowed,
	       max_insertionlen,max_deletionlen,overall_end_distance,
	       /*plusp*/false,genestrand,paired_end_p,first_read_p,
	       intlistpool,univcoordlistpool,listpool,univdiagpool,
	       hitlistpool,level);

  return;
}


